package ru.yandex.direct.oneshot.oneshots.attach_wallet_to_user_campaigns;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueObjType;
import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueuePriority;
import ru.yandex.direct.core.entity.balance.model.BalanceNotificationInfo;
import ru.yandex.direct.core.entity.balance.service.BalanceInfoQueueService;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignBalanceService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.validation.defects.RightsDefects;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsRecord;
import ru.yandex.direct.dbutil.exception.RollbackException;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.UNDER_WALLET;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;


/**
 * Выполняет прикрепление кампаний к кошелькам для полученного списка clientIds
 * Выполнять только для небольшого количества clientIds (count <= 2000)
 * <p>
 * Прикрепление только если:
 * 1. У клиентов есть кошелек
 * 2. У клиентов есть хотя бы одна не прикрепленная к кошельку кампания
 * 3. Ни в одной не прикрепленной к кошельку кампании нет денег (campaigns.sum - sum_spent <= 0)
 * <p>
 * Отправка в баланс учитывает не все кейсы
 * По остальным кейсам стоит посмотреть в CampaignWithSendingToBalanceAddOperationSupport.
 * Для текущих задач недостающая логика из CampaignWithSendingToBalanceAddOperationSupport кажется лишней.
 * Так же стоит быть осторожным при отправке в баланс через метод createOrUpdateOrdersOnFirstCampaignCreation
 * большого количества кампаний
 */
@Component
@Multilaunch
@Approvers({"ppalex", "ssdmitriev", "mspirit", "dimitrovsd", "khuzinazat", "a-dubov", "maxlog", "gerdler"})
public class AttachWalletToClientCampaignsOneshot implements SimpleOneshot<InputData, Void> {
    private static final Logger logger = LoggerFactory.getLogger(AttachWalletToClientCampaignsOneshot.class);

    private static final int MAX_CLIENT_LIST_SIZE = 2000;
    private static final Set<CampaignsType> UNDER_WALLET_TYPES = StreamEx.of(UNDER_WALLET)
            .map(CampaignType::toSource)
            .toSet();

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final CampaignBalanceService campaignBalanceService;
    private final CampaignTypedRepository campaignTypedRepository;
    private final BalanceInfoQueueService balanceInfoQueueService;

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        ItemValidationBuilder<InputData, Defect> builder = ItemValidationBuilder.of(inputData);
        builder.item(inputData.getClientIds(), "clientIds")
                .check(notNull())
                .check(notEmptyCollection())
                .check(maxListSize(MAX_CLIENT_LIST_SIZE))
                .checkBy(this::validateClientIds, When.isValid());
        return builder.getResult();
    }

    private ValidationResult<List<Long>, Defect> validateClientIds(List<Long> clientIds) {
        var builder = ListValidationBuilder.of(clientIds, Defect.class)
                .checkEach(notNull())
                .checkEach(validId())
                .checkEach(unique());

        if (!builder.getResult().hasAnyErrors()) {
            Map<Integer, List<Long>> shardToClientIds = getShardToClientIds(clientIds);

            Set<Long> clientIdsWithoutUnderWalletCampaignsAndMoney =
                    getClientIdsWithoutUnderWalletCampaignsAndMoney(shardToClientIds, clientIds);
            var clientIdToClient = StreamEx.of(clientService.massGetClient(listToSet(clientIds, ClientId::fromLong)))
                    .mapToEntry(Client::getClientId)
                    .invert()
                    .toMap();

            builder
                    .checkEach(fromPredicate(clientId -> doesClientExistAndWithoutUnderWalletCampaignsAndMoney(clientId,
                            clientIdsWithoutUnderWalletCampaignsAndMoney), CommonDefects.invalidValue()))
                    .checkEach(fromPredicate(clientId -> canClientHaveSharedAccount(clientIdToClient.get(clientId)),
                            RightsDefects.noRights()));
        }

        return builder.getResult();
    }

    /**
     * Проверка, что клиент существует и подходит для привязки ОС к кампаниям
     */
    private static boolean doesClientExistAndWithoutUnderWalletCampaignsAndMoney(
            Long clientId,
            Set<Long> clientIdsWithoutUnderWalletCampaigns) {
        if (!clientIdsWithoutUnderWalletCampaigns.contains(clientId)) {
            logger.error("Client {} does not exist, or does not have any campaigns, or does not have a wallet, " +
                    "or one of the campaigns has money", clientId);
            return false;
        }
        return true;
    }

    /**
     * Проверка, что у клиента может быть ОС
     */
    private static boolean canClientHaveSharedAccount(Client client) {
        if (Boolean.TRUE.equals(client.getSharedAccountDisabled())) {
            logger.error("Client {} cannot have a shared account", client.getClientId());
            return false;
        }
        return true;
    }

    @Autowired
    public AttachWalletToClientCampaignsOneshot(DslContextProvider dslContextProvider,
                                                ShardHelper shardHelper,
                                                ClientService clientService,
                                                CampaignBalanceService campaignBalanceService,
                                                CampaignTypedRepository campaignTypedRepository,
                                                BalanceInfoQueueService balanceInfoQueueService) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.campaignBalanceService = campaignBalanceService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.balanceInfoQueueService = balanceInfoQueueService;
    }

    @Nullable
    @Override
    public Void execute(InputData inputData, Void prevState) {
        Preconditions.checkNotNull(inputData);

        Map<Integer, List<Long>> shardToClientIds = getShardToClientIds(inputData.getClientIds());

        for (Map.Entry<Integer, List<Long>> entry : shardToClientIds.entrySet()) {
            Integer shard = entry.getKey();
            List<Long> clientIds = entry.getValue();

            for (Long clientId : clientIds) {
                updateWalletForClientCampaigns(shard, ClientId.fromLong(clientId));
            }
        }
        return null;
    }

    /**
     * Прикрепляем кошельки к кампаниям клиента {@param clientId}
     */
    private void updateWalletForClientCampaigns(int shard, ClientId clientId) {
        List<CommonCampaign> campaigns = new ArrayList<>();
        List<CommonCampaign> wallets = new ArrayList<>();
        var userId = new AtomicLong();

        try {
            dslContextProvider.ppcTransaction(shard, configuration -> {
                var dslContext = configuration.dsl();

                List<CommonCampaign> clientWalletAndCampaigns = getClientCampaigns(dslContext, clientId);

                campaigns.addAll(filterList(clientWalletAndCampaigns, c -> !CampaignType.WALLET.equals(c.getType())));
                wallets.addAll(filterList(clientWalletAndCampaigns, c -> CampaignType.WALLET.equals(c.getType())));
                Preconditions.checkState(wallets.size() == 1);
                CommonCampaign wallet = wallets.get(0);

                userId.set(campaigns.stream()
                        .map(CommonCampaign::getUid)
                        .findFirst()
                        .orElseThrow());

                var appliedChanges = StreamEx.of(campaigns)
                        .map(campaign -> {
                            logger.info("shard: {}, client: {}, cid: {}. Update CAMPAIGNS.WALLET_CID old: {} new: {}",
                                    shard, clientId, campaign.getId(), campaign.getWalletId(), wallet.getId());

                            return new ModelChanges<>(campaign.getId(), CommonCampaign.class)
                                    .process(wallet.getId(), CommonCampaign.WALLET_ID)
                                    .process(CampaignStatusBsSynced.NO, CommonCampaign.STATUS_BS_SYNCED)
                                    .process(LocalDateTime.now(), CommonCampaign.LAST_CHANGE)
                                    .applyTo(campaign);
                        })
                        .toList();

                // Прикрепляем кампании к кошельку
                int countOfUpdatedCampaigns = updateCampaigns(dslContext, appliedChanges);
                if (countOfUpdatedCampaigns != appliedChanges.size()) {
                    logger.warn("shard: {}, client {}. Some campaigns have not been changed", shard, clientId);
                    throw new RollbackException();
                }
            });
        } catch (RollbackException e) {
            logger.error("shard: {}, client: {}. Failed to change campaigns", shard, clientId);
            return;
        }

        CommonCampaign wallet = wallets.get(0);

        // Меняем ОС перед отправкой в баланс
        campaigns.forEach(c -> c.setWalletId(wallet.getWalletId()));

        sendCampaignsToBalance(shard, clientId, userId.get(), campaigns, wallet);
    }

    /**
     * Отправка кампаний в баланс
     */
    private void sendCampaignsToBalance(int shard,
                                        ClientId clientId,
                                        Long userId,
                                        List<CommonCampaign> campaigns,
                                        CommonCampaign wallet) {
        logger.warn("shard: {}, client: {}. Send {} campaigns and a wallet to balance",
                shard, clientId, campaigns.size());
        boolean creatingOrderInBalanceForWalletIsSuccessful = campaignBalanceService
                .createOrUpdateOrdersOnFirstCampaignCreation(userId, null, null, List.of(wallet));
        boolean creatingOrderInBalanceForCampaignsIsSuccessful = campaignBalanceService
                .createOrUpdateOrdersOnFirstCampaignCreation(userId, null, null, campaigns);

        if (!creatingOrderInBalanceForWalletIsSuccessful || !creatingOrderInBalanceForCampaignsIsSuccessful) {
            logger.warn("shard: {}, client: {}. Add {} campaigns to balance_info_queue",
                    shard, clientId, campaigns.size());
            addToBalanceQueue(dslContextProvider.ppc(shard), userId, ListUtils.union(campaigns, List.of(wallet)));
        }
    }

    /**
     * Добавление кампаний в очередь переотправки в Баланс
     */
    private void addToBalanceQueue(DSLContext context, Long userId, List<CommonCampaign> campaigns) {
        List<BalanceNotificationInfo> balanceNotificationInfos = StreamEx.of(campaigns)
                .map(campaign -> new BalanceNotificationInfo()
                        .withCidOrUid(campaign.getId())
                        .withObjType(BalanceInfoQueueObjType.CID)
                        .withPriority(BalanceInfoQueuePriority.PRIORITY_CAMPS_ON_ENABLE_WALLET)
                        .withOperatorUid(userId))
                .toList();

        balanceInfoQueueService.addToBalanceInfoQueue(context, balanceNotificationInfos);
    }

    private Map<Integer, List<Long>> getShardToClientIds(List<Long> clientIds) {
        return EntryStream.of(shardHelper.getShardsByClientIds(clientIds))
                .nonNullValues()
                .invert()
                .grouping();
    }

    /**
     * Возвращает id клиентов из списка {@param clientIds}, у которых есть кошелек и кампании без привязки к кошельку
     * и все эти кампании без денег (sum - sum_spent <= 0)
     */
    private Set<Long> getClientIdsWithoutUnderWalletCampaignsAndMoney(Map<Integer, List<Long>> shardToClientIds,
                                                                      List<Long> clientIds) {
        Field<BigDecimal> campaignsSumRest = CAMPAIGNS.SUM.minus(CAMPAIGNS.SUM_SPENT);
        return EntryStream.of(shardToClientIds)
                .mapKeyValue((shard, clientIdsInShard) ->
                        dslContextProvider.ppc(shard)
                                .selectDistinct(CLIENTS.CLIENT_ID)
                                .from(CAMPAIGNS)
                                .join(CLIENTS).on(CLIENTS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                                .where(CAMPAIGNS.WALLET_CID.eq(0L))
                                .and(CAMPAIGNS.TYPE.in(UNDER_WALLET_TYPES))
                                .and(CAMPAIGNS.CLIENT_ID.in(DSL.selectDistinct(CAMPAIGNS.CLIENT_ID)
                                        .from(CAMPAIGNS)
                                        .where(CAMPAIGNS.CLIENT_ID.in(clientIdsInShard))
                                        .and(CAMPAIGNS.TYPE.eq(CampaignsType.wallet))
                                        .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))))
                                .and(CAMPAIGNS.CLIENT_ID.notIn(DSL.selectDistinct(CAMPAIGNS.CLIENT_ID)
                                        .from(CAMPAIGNS)
                                        .where(CAMPAIGNS.CLIENT_ID.in(clientIdsInShard))
                                        .and(CAMPAIGNS.TYPE.ne(CampaignsType.wallet))
                                        .and(CAMPAIGNS.WALLET_CID.eq(0L))
                                        .and(campaignsSumRest.greaterThan(Currencies.EPSILON))
                                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))))
                                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                                .fetchSet(CLIENTS.CLIENT_ID)
                )
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
    }

    /**
     * Возвращает список кампаний клиента, которые не подключены к кошельку + сам кошелек
     */
    private List<CommonCampaign> getClientCampaigns(DSLContext dslContext, ClientId clientId) {
        Set<CampaignType> campaignTypes = new HashSet<>(UNDER_WALLET);
        campaignTypes.add(CampaignType.WALLET);

        return StreamEx.of(campaignTypedRepository.getTypedCampaigns(dslContext, clientId, campaignTypes))
                .map(c -> (CommonCampaign) c)
                .filter(c -> Long.valueOf(0L).equals(c.getWalletId()))
                .toList();
    }

    private int updateCampaigns(DSLContext dslContext, Collection<AppliedChanges<CommonCampaign>> appliedChanges) {
        JooqUpdateBuilder<CampaignsRecord, CommonCampaign> updateBuilder =
                new JooqUpdateBuilder<>(CAMPAIGNS.CID, appliedChanges);

        updateBuilder.processProperty(CommonCampaign.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE);
        updateBuilder.processProperty(CommonCampaign.STATUS_BS_SYNCED, CAMPAIGNS.STATUS_BS_SYNCED,
                CampaignStatusBsSynced::toSource);
        updateBuilder.processProperty(CommonCampaign.WALLET_ID, CAMPAIGNS.WALLET_CID);

        return dslContext.update(CAMPAIGNS)
                .set(updateBuilder.getValues())
                .where(CAMPAIGNS.CID.in(updateBuilder.getChangedIds()))
                .execute();
    }
}
