package ru.yandex.direct.oneshot.oneshots.autoretargeting

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupService
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupService.ComplexAdGroupFetchParams
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.uc.UcCampaignService
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.retargeting.service.uc.UcRetargetingConditionService
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.oneshot.base.SimpleYtOneshot
import ru.yandex.direct.oneshot.base.YtInputData
import ru.yandex.direct.oneshot.base.YtState
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import kotlin.math.min

class InputData(
    var chunkSize: Int
) : YtInputData()

/**
 * Ваншот обновления (создания/удаления/изменения) авторетаргетинга для UC-кампаний.
 *
 * Принимает на вход JSON
 * ```json
 * {
 *      "tablePath": "//home/direct/test/khuzinazat/campaigns_without_autoretargeting",
 *      "ytCluster": "hahn",
 *      "chunkSize": 10
 * }
 * ```
 * где
 * * `tablePath` - путь до таблички в YT с данными по кампаниям, для которых нужно обновить авторетаргетинг,
 * данные вида `<shard (int), cid (long)>`;
 * * `ytCluster` - кластер YT;
 * * `chunkSize` - размер чанка, для которого применяется ваншот.
 *
 * Пример получения данных по кампаниям на основе АБ экспериментов
 * [в тикете](https://st.yandex-team.ru/DIRECT-143704#607e9b6b4bf61b114ae85ee0)
 */
@Component
@Multilaunch
@Approvers("bratgrim", "gerdler", "dimitrovsd", "khuzinazat")
class UcAutoRetargetingOneshot @Autowired constructor(
    ytProvider: YtProvider,
    private val adGroupRepository: AdGroupRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val rbacService: RbacService,
    private val clientGeoService: ClientGeoService,
    private val ucCampaignService: UcCampaignService,
    private val complexAdGroupService: ComplexAdGroupService,
    private val recentStatisticsService: RecentStatisticsService,
    private val ucRetargetingConditionService: UcRetargetingConditionService
) : SimpleYtOneshot<InputData, YtState>(ytProvider) {
    companion object {
        private val logger = LoggerFactory.getLogger(UcAutoRetargetingOneshot::class.java)
    }

    override fun execute(inputData: InputData, prevState: YtState?): YtState? {
        val ytCluster = YtCluster.parse(inputData.ytCluster)
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator = ytProvider.getOperator(ytCluster)

        if (prevState == null) {
            logger.info("First iteration!")
            return YtState().apply {
                nextRow = 0L
                totalRowCount = ytOperator.readTableRowCount(ytTable)
            }
        }

        val rowCount = prevState.totalRowCount
        val startRow = prevState.nextRow
        val endRow = min(prevState.nextRow + inputData.chunkSize, rowCount)
        if (startRow >= rowCount) {
            logger.info("Last iteration, last processed row: {}, total rows: {}", startRow, rowCount)
            return null
        }
        try {
            val tableRow = UcAutoRetargetingTableRow()
            val campaignIdsByShard = mutableMapOf<Int, MutableSet<Long>>()
            ytOperator.readTableByRowRange(
                ytTable, {
                row: UcAutoRetargetingTableRow ->
                campaignIdsByShard.computeIfAbsent(row.shard) { mutableSetOf() }
                    .add(row.campaignId)
                }, tableRow, startRow, endRow
            )
            val processedCampaignIds = mutableListOf<Long>()
            campaignIdsByShard.forEach { (shard, campaignIds) ->
                logger.info("Start processing on shard={}, campaignIds={}", shard, campaignIds)
                val campaigns = campaignTypedRepository.getTypedCampaigns(shard, campaignIds)
                val adGroupIdsByCampaignIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds)
                campaigns.forEach {
                    val processed = processCampaign(it, adGroupIdsByCampaignIds)
                    if (processed) {
                        processedCampaignIds.add(it.id)
                    }
                }
            }
            val campaignIdsSize = campaignIdsByShard.values.flatten().size
            logger.info("Successfully processed {} campaigns of {} ids from row {} to {}",
                processedCampaignIds.size, campaignIdsSize, startRow, endRow)
        } catch (e: RuntimeException) {
            logger.error("Caught exception: {}", e.message)
            throw e
        }
        return YtState().apply {
            nextRow = endRow
            totalRowCount = rowCount
        }
    }

    private fun processCampaign(campaign : BaseCampaign, adGroupIdsByCampaignIds : Map<Long, List<Long>>): Boolean {
        logger.info("Start processing campaign={}", campaign.id)
        if (campaign !is TextCampaign || !AvailableCampaignSources.isUC(campaign.source)) {
            logger.error("Campaign={} is not UC campaign", campaign.id)
            return false
        }
        if (campaign.statusArchived) {
            logger.error("Campaign={} is archived", campaign.id)
            return false
        }
        val adGroupIds = adGroupIdsByCampaignIds[campaign.id]
        if (adGroupIds == null || adGroupIds.isEmpty()) {
            logger.warn("Ad group not found for campaign={}", campaign.id)
            return false
        }

        val clientId = ClientId.fromLong(campaign.clientId)
        val adGroupId = adGroupIds.minOrNull()

        val clientUid = rbacService.getChiefByClientId(clientId)
        val uidAndClientId = UidAndClientId.of(clientUid, clientId)
        val complexAdGroup = complexAdGroupService.getComplexAdGroups(
            clientUid, uidAndClientId, listOf(adGroupId),
            ComplexAdGroupFetchParams.fetchAll()
                .withFetchKeywords(false)
                .withFetchRelevanceMatches(false)
                .withFetchOfferRetargetings(false)
                .withFetchRetargetingConditions(false)
                .withFetchBidModifiers(false)).firstOrNull()
        if (complexAdGroup == null) {
            logger.error("Complex ad group not found by clientUid={}, clientId={}, adGroupId={}",
                clientUid, clientId, adGroupId)
            return false
        }

        val metrikaCounters = campaign.metrikaCounters?.map { it.toInt() }
        val goalIds = getGoalIds(campaign.strategy.strategyData.goalId, campaign.meaningfulGoals)
        logger.info("Get autoretargeting condition for campaign={} with metrikaCounters={}, goalIds={}, adGroupId={}",
            campaign.id, metrikaCounters, goalIds, adGroupId)
        val autoRetargetingCondition = ucRetargetingConditionService.getAutoRetargetingCondition(
            clientId, metrikaCounters, goalIds, adGroupId)

        val isCampaignWithoutAutoRetargeting = complexAdGroup.targetInterests.isNullOrEmpty()
        if (autoRetargetingCondition == null && isCampaignWithoutAutoRetargeting) {
            logger.info("Campaign ${campaign.id} has no autoretargeting " +
                "and computed autoretargeting is null, do not update campaign")
            return false;
        }

        complexAdGroup.apply {
            retargetingCondition = autoRetargetingCondition
            targetInterests = if (autoRetargetingCondition != null) listOf(autoRetargetingCondition.targetInterest) else null
            complexBanners = null // не обновляем баннера
        }
        val autoPriceParams = ShowConditionAutoPriceParams(
            ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null), recentStatisticsService)
        val geoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        val vr = ucCampaignService.updateGroup(complexAdGroup,
            setOf(ComplexTextAdGroup.RETARGETING_CONDITION, ComplexTextAdGroup.TARGET_INTERESTS),
            geoTree, autoPriceParams, clientUid, uidAndClientId).validationResult
        if (vr != null && vr.hasAnyErrors()) {
            val vrErrors = vr.flattenErrors()
                .map { it.toString() }
                .joinToString("; ")
            val operation = when {
                autoRetargetingCondition == null -> "deleting"
                isCampaignWithoutAutoRetargeting -> "creating"
                else -> "updating"
            }
            logger.error("Error while $operation autoretargeting " +
                "for campaign=${campaign.id} and adGroupId=$adGroupId: $vrErrors")
            return false
        }
        val operation = when {
            autoRetargetingCondition == null -> "deleted"
            isCampaignWithoutAutoRetargeting -> "created"
            else -> "updated"
        }
        logger.info("Successfully $operation autoretargeting for campaign=${campaign.id}")
        return true
    }

    private fun getGoalIds(goalId : Long?, meaningfulGoals: List<MeaningfulGoal>?): List<Long>? {
        if (goalId == null) {
            return null
        }
        return if (goalId == CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)
            meaningfulGoals?.map { it.goalId }
        else
            listOf(goalId)
    }
}
