package ru.yandex.direct.oneshot.oneshots.bannerperpage

import org.slf4j.LoggerFactory.getLogger
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan
import ru.yandex.direct.validation.result.Defect

data class InputData(
        var maxValue: Long = 30L,
        var newValue: Long = 5L
)

data class State(
        val lastCid: Long
)

@Component
@Multilaunch
@Approvers("maxlog", "adubinkin")
class BannerPerPageOneshot @Autowired constructor(
        private val dsl: DslContextProvider
) : ShardedOneshot<InputData, State?> {
    companion object {
        private val logger = getLogger(BannerPerPageOneshot::class.java)
        private const val ONE_STEP_LIMIT = 10_000L
    }

    override fun validate(inputData: InputData) =
            ItemValidationBuilder.of(inputData, Defect::class.java).apply {
                item(inputData.maxValue, "maxValue")
                        .check(notNull())
                        .check(notLessThan(1L))
                item(inputData.newValue, "newValue")
                        .check(notNull())
                        .check(notLessThan(0L))
            }.result

    override fun execute(inputData: InputData, prevState: State?, shard: Int): State? {
        val lastCid = prevState?.lastCid ?: 0
        val cids = getCidsForChanging(shard, lastCid, inputData.maxValue)
        if (cids.isEmpty()) {
            logger.info("There are no rows for changing, shard {}", shard)
            return null
        }
        logger.info("Migrating {} rows to new value, shard {}", cids.size, shard)
        val rowsNumber = updateBannerPerPage(shard, cids, inputData.newValue)
        logger.info("The number of updated records: {}, shard {}", rowsNumber, shard)
        return State(cids.last())
    }

    fun getCidsForChanging(shard: Int, lastCid: Long, maxValue: Long) =
            dsl.ppc(shard)
                    .select(CAMP_OPTIONS.CID)
                    .from(CAMP_OPTIONS)
                    .where(
                            CAMP_OPTIONS.CID.greaterThan(lastCid)
                                    .and(CAMP_OPTIONS.BANNERS_PER_PAGE.greaterThan(maxValue))
                    )
                    .orderBy(CAMP_OPTIONS.CID.asc())
                    .limit(ONE_STEP_LIMIT)
                    .fetch(CAMP_OPTIONS.CID)

    private fun updateBannerPerPage(shard: Int, cids: List<Long>, newValue: Long) =
            dsl.ppc(shard)
                    .update(CAMP_OPTIONS)
                    .set(CAMP_OPTIONS.BANNERS_PER_PAGE, newValue)
                    .where(CAMP_OPTIONS.CID.`in`(cids))
                    .execute()

}
