package ru.yandex.direct.oneshot.oneshots.bidmodifiers

import org.jooq.DSLContext
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.singlevalue.BidModifierDesktopSingleValueTypeSupport
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.base.ShardedOneshotWithoutInput
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import java.util.stream.Collectors

@Component
@Multilaunch
@Approvers("a-dubov", "bakin")
class ChangeDesktopOnlyToDesktopForCPM @Autowired  constructor(
    private val dslContextProvider: DslContextProvider,
    private val desktopTypeSupport: BidModifierDesktopSingleValueTypeSupport,
    private val campaignRepository: CampaignRepository)
    : ShardedOneshotWithoutInput() {


    companion object {
        private val logger = LoggerFactory.getLogger(ChangeDesktopOnlyToDesktopForCPM::class.java)
    }

    override fun execute(shard: Int) {
        logger.info(String.format("Shard: %d: ChangeDesktopOnlyToDesktopForCPM START", shard))

        val dslContext = dslContextProvider.ppc(shard)

        val multipliers = getDataToChange(dslContext)
        deleteDesktopOnlyModifiers(dslContext, multipliers.values.map { data -> data.id })
        insertDesktopModifiers(dslContext, multipliers)

        // переотправляем затронутые кампании в БК
        resendToBs(dslContext, multipliers.values.map{ data -> data.cid }.toCollection(mutableSetOf()))
        logger.info(String.format("Shard: %d - updated %d records", shard, multipliers.size))
        logger.info(String.format("Shard: %d: ChangeDesktopOnlyToDesktopForCPM FINISH", shard))
    }

    // достаем id корректировок в которых требуются изменения
    private fun getDataToChange(dslContext: DSLContext): Map<Long, DataObject> {
        return dslContext
            .select(
                Tables.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID,
                Tables.HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT,
                Tables.CAMPAIGNS.CID,
                Tables.HIERARCHICAL_MULTIPLIERS.PID,
                Tables.CAMPAIGNS.UID
            )
            .from(Tables.HIERARCHICAL_MULTIPLIERS)
            .join(Tables.CAMPAIGNS).using(Tables.HIERARCHICAL_MULTIPLIERS.CID)
            .where(Tables.CAMPAIGNS.TYPE.equal(CampaignsType.cpm_banner)
                .and(DSL.or(Tables.HIERARCHICAL_MULTIPLIERS.TYPE.eq(HierarchicalMultipliersType.desktop_only_multiplier))))
            .fetchMap(
                { it.getValue(Tables.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID) },
                { DataObject(
                    it.getValue(Tables.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID),
                    it.getValue(Tables.HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT),
                    it.getValue(Tables.HIERARCHICAL_MULTIPLIERS.CID),
                    it.getValue(Tables.CAMPAIGNS.UID),
                    it.getValue(Tables.HIERARCHICAL_MULTIPLIERS.PID)
                ) })
    }

    // добавляем необходимые корректировки
    private fun deleteDesktopOnlyModifiers(dslContext: DSLContext,
                                          ids: Collection<Long>) {
        dslContext
            .deleteFrom(Tables.HIERARCHICAL_MULTIPLIERS)
            .where(Tables.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.`in`(ids))
            .execute()
    }

    // добавляем необходимые корректировки
    private fun  insertDesktopModifiers(dslContext: DSLContext,
                                        dataList: Map<Long, DataObject>) {

        dataList.values
            .groupBy { data -> data.uid }
            .forEach {
                val modifiers = it.value.map { data -> BidModifierDesktop()
                    .withType(BidModifierType.DESKTOP_MULTIPLIER)
                    .withCampaignId(data.cid)
                    .withAdGroupId(data.pid)
                    .withDesktopAdjustment(BidModifierDesktopAdjustment()
                        .withPercent(data.value.toInt()))}
                desktopTypeSupport.prepareSystemFields(modifiers)
                desktopTypeSupport.add(dslContext, modifiers, it.key)
            }
    }

    private fun resendToBs(dslContext: DSLContext,
                           cids: Collection<Long>) {
        campaignRepository.updateStatusBsSynced(dslContext, cids, StatusBsSynced.NO)
        logger.info(String.format("Sent to BS: %d campaigns: %s",cids.size,
                cids.stream().map{it.toString()}.collect(Collectors.joining(", "))))
    }

}

class DataObject(
    val id: Long,
    val value: Long,
    val cid: Long,
    val uid: Long,
    val pid: Long
)
