package ru.yandex.direct.oneshot.oneshots.bidmodifiers

import org.jooq.DSLContext
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment
import ru.yandex.direct.core.entity.bidmodifier.InventoryType
import ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.INVENTORY_ADJUSTMENT_MAPPER
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.dbschema.ppc.Tables.*
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbschema.ppc.enums.InventoryMultiplierValuesInventoryType
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.jooqmapperhelper.UpdateHelper
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.base.ShardedOneshotWithoutInput
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.utils.FunctionalUtils.filterList
import ru.yandex.direct.utils.FunctionalUtils.mapList
import java.math.BigDecimal
import java.time.LocalDateTime
import java.util.stream.Collectors.joining

@Component
@Multilaunch
@Approvers("andreypav", "hrustyashko")
class EnrichToInBannerInAppInventoryMultiplierPairOneShot constructor(
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
    private val campaignRepository: CampaignRepository)
    : ShardedOneshotWithoutInput() {

    companion object {
        private val logger = LoggerFactory.getLogger(EnrichToInBannerInAppInventoryMultiplierPairOneShot::class.java)
    }

    override fun execute(shard: Int) {
        logger.info(String.format("Shard: %d: EnrichToInBannerInAppInventoryMultiplierPairOneShot START", shard))

        val dslContext = dslContextProvider.ppc(shard)

        val subInventoryData = getSubInventoryData(dslContext)
        fillData(dslContext, subInventoryData)

        // Сохраняем в БД пары для корректировок
        insertInventoryModifiers(dslContext, subInventoryData)
        updateInventoryModifiers(dslContext, subInventoryData)
        // переотправляем затронутые кампании в БК
        val affectedModifiers = resendToBs(dslContext, subInventoryData)
        logger.info(String.format("Shard: %d - updated %d records", shard, affectedModifiers))
        logger.info(String.format("Shard: %d: EnrichToInBannerInAppInventoryMultiplierPairOneShot FINISH", shard))
    }

    // достаем id корректировок в которых требуются изменения, а также данные, которые подскажут какие именно изменения
    private fun getSubInventoryData(dslContext: DSLContext): Map<Long, InventoryCommonDataObject> {
        return dslContext
            .select(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID,
                DSL.sum(DSL.iif(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE.eq(
                    InventoryMultiplierValuesInventoryType.inbanner), 1, 0)).`as`("inbanner"),
                DSL.sum(DSL.iif(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE.eq(
                    InventoryMultiplierValuesInventoryType.inapp), 1, 0)).`as`("inapp"),
                CAMPAIGNS.CID)
            .from(INVENTORY_MULTIPLIER_VALUES)
            .join(HIERARCHICAL_MULTIPLIERS).using(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)
            .join(CAMPAIGNS).using(HIERARCHICAL_MULTIPLIERS.CID)
            .where(CAMPAIGNS.TYPE.equal(CampaignsType.cpm_banner)
                .and(DSL.or(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE.eq(InventoryMultiplierValuesInventoryType.inbanner))
                    .or(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE.eq(InventoryMultiplierValuesInventoryType.inapp))))
            .groupBy(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)
            .fetchMap(
                { it.getValue(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID) },
                {
                val inbanner:Boolean = it.getValue("inbanner") as BigDecimal == BigDecimal.ONE
                val inapp:Boolean = it.getValue("inapp") as BigDecimal == BigDecimal.ONE
                val type: InventorySelectType;
                if (inbanner && inapp) {
                    type = InventorySelectType.BOTH
                } else if (inbanner) {
                    type = InventorySelectType.INBANNER
                } else
                    type = InventorySelectType.INAPP

                InventoryCommonDataObject(
                    it.getValue(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID), type,
                    it.getValue(CAMPAIGNS.CID))
            })
    }

    // по выбранным  id корректировки извлекаем из БД нужные для дальнейшей работы данные
    private fun fillData(dslContext: DSLContext, subInventoryData : Map<Long, InventoryCommonDataObject>) {
        dslContext.select(
                INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID,
                INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID,
                INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE,
                INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT)
            .from(INVENTORY_MULTIPLIER_VALUES)
            .join(HIERARCHICAL_MULTIPLIERS).using(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)
            .where(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID.`in`(subInventoryData.keys)
                .and(DSL.or(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE.eq(InventoryMultiplierValuesInventoryType.inbanner))
                    .or(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE.eq(InventoryMultiplierValuesInventoryType.inapp))))
            .fetch {
                val element = subInventoryData
                    .get(it.getValue(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID))
                if (element?.type == InventorySelectType.BOTH) {
                    val type = it.getValue(INVENTORY_MULTIPLIER_VALUES.INVENTORY_TYPE)
                        as InventoryMultiplierValuesInventoryType
                    // Если корректировки две, то проставляем только процент от inapp, так как он главнее
                    if (type == InventoryMultiplierValuesInventoryType.inapp) {
                        element.percent = it.getValue(INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT).toInt()
                    }
                    // теоретически может по какой-то причине inventoryValuesId остаться null,
                    // далее отсеем их, если вдруг найдутся такие.
                    if (type == InventoryMultiplierValuesInventoryType.inbanner) {
                        element.oldPercent = it.getValue(INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT).toInt()
                        element.inventoryValuesId = it.getValue(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID)
                    }
                } else {
                    // Если корректировок только одна, тогда берем тот процент, что есть
                    element?.percent = it.getValue(INVENTORY_MULTIPLIER_VALUES.MULTIPLIER_PCT).toInt()
                    element?.inventoryValuesId = it.getValue(INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID)
                }
            }
    }

    // добавляем необходимые корректировки
    private fun  insertInventoryModifiers(dslContext: DSLContext,
                                          commonDataList: Map<Long, InventoryCommonDataObject>) {
        // Отрабатываем вставку второй корректировки для пары
        val insertHelper = InsertHelper(dslContext, INVENTORY_MULTIPLIER_VALUES)
        val insertValues: List<InventoryCommonDataObject> = filterList(commonDataList.values,
            {data -> data.type != InventorySelectType.BOTH})

        val ids: List<Long> = shardHelper.generateHierarchicalMultiplierIds(insertValues.size)
        logger.info(String.format("For add: %d modifiers", insertValues.size))
        insertValues.forEachIndexed { index: Int, item: InventoryCommonDataObject ->
            val type = if (item.type == InventorySelectType.INBANNER) InventoryType.INAPP else InventoryType.INBANNER
            val adjustment: BidModifierInventoryAdjustment = BidModifierInventoryAdjustment()
                .withId(ids.get(index))
                .withInventoryType(type)
                .withPercent(item.percent)
                .withLastChange(LocalDateTime.now())
            insertHelper.add(INVENTORY_ADJUSTMENT_MAPPER, adjustment)
            insertHelper.set(INVENTORY_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID,
                item.hierarchicalMultiplierId)
            insertHelper.newRecord()
            logger.info(String.format("Added: inventory_multiplier_value_id: %d, percent: %d, type: %s",
                adjustment.id, adjustment.percent, adjustment.inventoryType.name))

        }
        insertHelper.executeIfRecordsAdded()
    }

    // редактируем необходимые корректировки
    private fun updateInventoryModifiers(dslContext: DSLContext,
                                         commonDataList: Map<Long, InventoryCommonDataObject>) {
        val updateValues: List<InventoryCommonDataObject> = filterList(commonDataList.values,
            {data -> data.type == InventorySelectType.BOTH
                && data.inventoryValuesId != null
                && data.percent != data.oldPercent})

        // Отрабатываем корректировки в которых есть обе inapp и inbanner, чтобы выставить одинаковый процент
        logger.info(String.format("For edit: %d modifiers", updateValues.size))
        updateValues.forEach { item: InventoryCommonDataObject ->
            val modelChanges: ModelChanges<BidModifierInventoryAdjustment> =
                ModelChanges(item.inventoryValuesId, BidModifierInventoryAdjustment::class.java)
            modelChanges.process(item.percent, BidModifierInventoryAdjustment.PERCENT)
            modelChanges.process(LocalDateTime.now(), BidModifierInventoryAdjustment.LAST_CHANGE)
            val adjustment: BidModifierInventoryAdjustment = BidModifierInventoryAdjustment()
                .withInventoryType(InventoryType.INBANNER)
                .withId(item.inventoryValuesId)
                .withPercent(null)
                .withLastChange(null)
            val ac: AppliedChanges<BidModifierInventoryAdjustment> = modelChanges.applyTo(adjustment)

            UpdateHelper(dslContext, INVENTORY_MULTIPLIER_VALUES.INVENTORY_MULTIPLIER_VALUE_ID)
                .processUpdate(INVENTORY_ADJUSTMENT_MAPPER, ac)
                .execute()
            logger.info(String.format("Updated: inventory_multiplier_value_id: %d, percent: %d",
                adjustment.id, adjustment.percent))
        }
    }
    private fun resendToBs(dslContext: DSLContext,
                           commonDataList: Map<Long, InventoryCommonDataObject>): Int {
        val affectedModifiers: List<InventoryCommonDataObject> =
            filterList(commonDataList.values) {it.percent != it.oldPercent}

        val affectedCampaigns: List<Long> = mapList(affectedModifiers){ it.cid }
        if (!affectedCampaigns.isEmpty()) {
            campaignRepository.updateStatusBsSynced(dslContext, affectedCampaigns, StatusBsSynced.NO)
            logger.info(String.format("Sent to BS: %d campaigns: %s",affectedCampaigns.size,
                affectedCampaigns.stream().map{it.toString()}.collect(joining(", "))))
        }
        return affectedModifiers.size;
    }
}

class InventoryCommonDataObject(
    val hierarchicalMultiplierId: Long,
    val type: InventorySelectType,
    val cid: Long) {
    var inventoryValuesId: Long = 0
        get() = field
        set(value) {
            field = value
        }
    var percent: Int = 0
        get() = field
        set(value) {
            field = value
        }
    var oldPercent: Int = 0
        get() = field
        set(value) {
            field = value
        }
}

enum class InventorySelectType {
    INBANNER,
    INAPP,
    BOTH
}
