package ru.yandex.direct.oneshot.oneshots.bids;

import java.util.ArrayList;
import java.util.HashSet;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.keyword.container.CampaignIdAndKeywordIdPair;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Component
@Multilaunch
@Approvers({"ppalex", "gerdler", "kuhtich"})
public class DeleteGrouplessBids implements SimpleOneshot<DeleteGrouplessBidsInputData, Void> {
    private static final Logger logger = LoggerFactory.getLogger(DeleteGrouplessBids.class);
    private final DslContextProvider dslContextProvider;
    private final KeywordRepository keywordRepository;
    private final AdGroupService adGroupService;
    private final KeywordService keywordService;
    private final ShardHelper shardHelper;

    @Autowired
    public DeleteGrouplessBids(DslContextProvider dslContextProvider, KeywordRepository keywordRepository,
            AdGroupService adGroupService, KeywordService keywordService, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.keywordRepository = keywordRepository;
        this.adGroupService = adGroupService;
        this.keywordService = keywordService;
        this.shardHelper = shardHelper;
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ValidationResult<DeleteGrouplessBidsInputData, Defect> validate(
            DeleteGrouplessBidsInputData inputData) {
        ItemValidationBuilder<DeleteGrouplessBidsInputData, Defect> ivb = ItemValidationBuilder.of(inputData);
        ivb.item(inputData.getClientId(), "clientId")
                .check(notNull())
                .check(validId(), When.isValid());
        ivb.item(inputData.getBidsIds(), "bidsIds")
                .check(notNull())
                .check(eachNotNull(), When.isValid())
                .check(Constraint.fromPredicate(
                            l -> l.stream().allMatch(not(Predicates.lessThan(1L))),
                            CommonDefects.validId()
                        ),
                        When.isValid());
        return ivb.getResult();
    }

    @Nullable
    @Override
    public Void execute(DeleteGrouplessBidsInputData inputData,
            @Nullable Void prevState) {
        var clientId = ClientId.fromLong(inputData.getClientId());
        var shard = shardHelper.getShardByClientIdStrictly(clientId);
        var keywords = keywordService.getKeywords(clientId, inputData.getBidsIds());
        if (keywords.isEmpty()) {
            logger.error("No bids ids to delete");
            return null;
        }
        if (keywords.size() < inputData.getBidsIds().size()) {
            logger.warn("Couldn't find some bids ids");
        }
        var adGroupIds = listToSet(
                adGroupService.getAdGroups(clientId, mapList(keywords, Keyword::getAdGroupId)),
                AdGroup::getId
        );
        var bidsIdsToDelete = new ArrayList<CampaignIdAndKeywordIdPair>();
        var bidsIdsToLog = new HashSet<Long>();
        for (var keyword : keywords) {
            if (adGroupIds.contains(keyword.getAdGroupId())) {
                logger.warn("Keyword {} belongs to an existing group {}", keyword.getId(), keyword.getAdGroupId());
            } else {
                bidsIdsToDelete.add(new CampaignIdAndKeywordIdPair(keyword.getCampaignId(), keyword.getId()));
                bidsIdsToLog.add(keyword.getId());
            }
        }
        if (bidsIdsToDelete.isEmpty()) {
            logger.error("All keywords belong to some groups");
            return null;
        }

        logger.info("Will delete these records: {}", filterList(keywords, k -> bidsIdsToLog.contains(k.getId())));
        keywordRepository.deleteKeywords(dslContextProvider.ppc(shard).configuration(), bidsIdsToDelete);

        return null;
    }
}
