package ru.yandex.direct.oneshot.oneshots.bsexport

import org.slf4j.LoggerFactory
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.common.models.BaseLogicObject
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class ResyncParam(
    val ytCluster: YtCluster,
    val tablePath: String
)

data class ResyncState(
    val nextRow: Long = 0L
)

abstract class ResyncTableRow<T>(
    columns: List<YtField<out Any>>
) : YtTableRow(columns) {
    abstract fun convert(): T
}

/**
 * Базовый ваншот для переотправки информации о баннерах/кампаниях в ess
 *
 * На вход принимает кластер и путь к таблице, где записано, что надо переотправить
 * Ваншот шардированный, поэтому если делается переотправка большого количества объектов, то стоит отсортировать строки по шарду
 * Читает строки пачками, размер пачки на данный момнт регулируется пропертей (см. в классах наследниках)
 * Если пачка для обработки непустая, то после итерации делается пауза, которая регулируется пропертей (тоже в классах наследниках)
 * Для переотправки большого количества объектов, стоит менять настройки в зависимости от нагрузки на контур ess
 * В будущем в этом месте планинруется автоматизация, ваншот сам будет следить за нагрузкой контура ess
 */
abstract class BaseResyncOneshot<T, U : ResyncTableRow<T>>(
    private val ytProvider: YtProvider,
    private val essClient: EssClient,
) : ShardedOneshot<ResyncParam, ResyncState> {
    companion object {
        private const val DEFAULT_CHUNK_SIZE = 10_000L
        private const val DEFAULT_RELAX_TIME = 300L
    }

    abstract val chunkSizeProperty: PpcProperty<Long>

    abstract val relaxTimeProperty: PpcProperty<Long>

    abstract val essLogicProcessName: String

    val logger = LoggerFactory.getLogger(this.javaClass)

    override fun validate(inputData: ResyncParam): ValidationResult<ResyncParam, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(ytProvider, vb, inputData.ytCluster, inputData.tablePath)
    }

    override fun execute(inputData: ResyncParam, prevState: ResyncState?, shard: Int): ResyncState? {
        val chunkSize = chunkSizeProperty.getOrDefault(DEFAULT_CHUNK_SIZE)
        val relaxTime = relaxTimeProperty.getOrDefault(DEFAULT_RELAX_TIME)
        val startRow = prevState?.nextRow ?: 0L
        val lastRow = startRow + chunkSize

        logger.info(
            "Shard=$shard: Start iteration with chunkSize=$chunkSize, " +
                    "relaxTime = $relaxTime, startRow=$startRow, lastRow=$lastRow{}"
        )

        val requests = getRequestsFromYt(inputData, startRow, lastRow, shard)

        val objects = getLogicObjects(shard, requests)

        if (objects.isNotEmpty()) {
            logger.info("Shard=$shard: Going to send ${objects.size} objects to ess")

            essClient.addLogicObjectsForProcessor(shard, essLogicProcessName, objects, true)

            logger.info("Shard=$shard: Iteration finished, sleep for $relaxTime seconds")
            Thread.sleep(relaxTime * 1000)
        } else {
            logger.info("Shard=$shard: No items found on iteration")
        }

        if (requests.size < chunkSize) {
            logger.info("Shard=$shard: Last iteration finished")
            essClient.clearLogicObjects(shard)
            return null
        }

        return ResyncState(lastRow)
    }

    abstract fun createEmptyYtRow(): U

    abstract fun getLogicObjects(shard: Int, requests: List<T>): List<BaseLogicObject>

    private fun getRequestsFromYt(inputData: ResyncParam, startRow: Long, lastRow: Long, shard: Int): List<T> {
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator = ytProvider.getOperator(inputData.ytCluster)
        val tableRow = createEmptyYtRow()
        val items = mutableListOf<T>()

        ytOperator.readTableByRowRange(
            ytTable,
            { row -> items.add(row.convert()) },
            tableRow,
            startRow,
            lastRow
        )
        logger.info("Shard=$shard: Got ${items.size} items from table")
        return items
    }
}
