package ru.yandex.direct.oneshot.oneshots.bsexport

import com.google.protobuf.Message
import org.slf4j.LoggerFactory
import ru.yandex.direct.bstransport.yt.repository.common.RowPosition
import ru.yandex.direct.bstransport.yt.service.BaseTableToQueueResyncService
import ru.yandex.direct.bstransport.yt.service.YtHashBorders
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.time.Instant

/**
 * Базовый ваншот для переотправки строк из таблицы Единой базы (//home/adv) в очередь Caesar (//home/adv/queues)
 * Ваншот шардированный, но к шарду базы на самом деле не привязан, используется для реализации параллельности,
 * тк в ваншотах нельзя параметризовывать чем то кроме шарда
 *
 * Каждый шард работает со своим диапозоном поля YtHash
 * Читает строки таблицы, преобразовывает их в объект формата protobuf, пишет в очередь
 * Как только количество прочитанных строк становится меньше лимита, ваншот прекращает работу(прочитал все)
 */
abstract class BaseTableToQueueOneshot<T1 : Message, T2 : RowPosition, T3>(
    private val tableToQueueResyncService: BaseTableToQueueResyncService<T1, T2>,
    private val shardSupport: ShardSupport
) : ShardedOneshot<Unit, T3> {
    companion object {
        private val logger = LoggerFactory.getLogger(BaseTableToQueueOneshot::class.java)
        private const val CHUNK_SIZE = 300_000
    }

    /**
     * Пропертя, хранящая количество строк за один запрос
     */
    abstract val chunkSizeProperty: PpcProperty<Int>

    /**
     * Получает первоночатные значения позиции для чтения. Передаеются границы YtHash, метод должен сфромировать границы,
     * включающие все ключевые колонки
     */
    abstract fun getPositionFromYtBorders(ytHashBorders: YtHashBorders): T2
    override fun validate(inputData: Unit): ValidationResult<Unit, Defect<Any>> {
        return ValidationResult.success(inputData)
    }

    fun resync(shard: Int, prevPosition: T2?): T2? {
        val ytHashBorders = tableToQueueResyncService.getYtHashBorders(shard - 1, shardSupport.availablePpcShards.size)
        if (ytHashBorders.isAbsent) {
            logger.info("There is no work for batch ${shard - 1}")
            return null
        }
        val position = prevPosition ?: getPositionFromYtBorders(ytHashBorders)

        val iterationStartTime = Instant.now().epochSecond
        val chunkSize = chunkSizeProperty.getOrDefault(CHUNK_SIZE)

        val objectsWithPosition = tableToQueueResyncService.resyncObjects(position, chunkSize, ytHashBorders.second)
        if (objectsWithPosition.objects.isEmpty()) return null

        logger.info("Handled ${objectsWithPosition.objects.size} objects, " +
            "current position = $position")

        val iterationDuration = Instant.now().epochSecond - iterationStartTime
        logger.info("Iteration duration = $iterationDuration seconds")
        return if (objectsWithPosition.objects.size < chunkSize) null
        else objectsWithPosition.position
    }
}
