package ru.yandex.direct.oneshot.oneshots.bsexport

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.bstransport.yt.repository.showconditions.BiddableShowConditionYtRepository
import ru.yandex.direct.core.bsexport.model.BsExportAbstractBid
import ru.yandex.direct.core.bsexport.repository.bids.BsExportBidsRepositoryInterface
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BidObjectType
import ru.yandex.direct.logicprocessor.processors.bsexport.bids.BidRepositoryDispatcher
import ru.yandex.direct.logicprocessor.processors.bsexport.bids.BiddableShowConditionsYtRecordMapper
import ru.yandex.direct.logicprocessor.processors.bsexport.bids.BsExportBiddableShowConditionsService
import ru.yandex.direct.oneshot.oneshots.bsexport.repository.BsExportCampaignsRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

data class BiddableShowConditionsUploadToYtParam(
    val shardsLastCid: Map<String, Long>?,
    val biddableConditionType: BidObjectType?,
)

data class BiddableShowConditionsUploadToYtState(
    val lastCampaignId: Long,
    val maxCampaignId: Long,
    val importedCount: Long
)

@Component
@Multilaunch
@Approvers("mspirit", "zakhar", "pema4")
@PausedStatusOnFail
// при нагрузке YT выдаёт ошибки, поэтому повторов так много
@Retries(100)
class BsExportBiddableShowConditionsUploadOneshot(
    private val mapper: BiddableShowConditionsYtRecordMapper,
    private val uploadYtRepository: BiddableShowConditionYtRepository,
    private val campaignsRepository: BsExportCampaignsRepository,
    private val bsExportBiddableShowConditionsService: BsExportBiddableShowConditionsService,
    private val bsExportBidRepositoryDispatcher: BidRepositoryDispatcher,
) : ShardedOneshot<BiddableShowConditionsUploadToYtParam, BiddableShowConditionsUploadToYtState?> {

    companion object {
        private val logger = LoggerFactory.getLogger(
            BsExportBiddableShowConditionsUploadOneshot::class.java
        )

        private const val CAMPAIGNS_PER_ITERATION = 100
        private const val BIDS_CHUNK = 10000
    }

    override fun validate(inputData: BiddableShowConditionsUploadToYtParam) = validateObject(inputData) {
        property(inputData::biddableConditionType) {
            check(CommonConstraints.inSet(BidObjectType.BID_TABLE_TYPES))
        }
    }

    override fun execute(
        inputData: BiddableShowConditionsUploadToYtParam?,
        prevState: BiddableShowConditionsUploadToYtState?,
        shard: Int
    ): BiddableShowConditionsUploadToYtState? {
        val lastCid = prevState?.lastCampaignId
            ?: inputData?.shardsLastCid?.get(shard.toString())
            ?: 0L
        val maxCid = prevState?.maxCampaignId ?: campaignsRepository.getMaxCid(shard) ?: -1
        logger.info("start from cid: {}", lastCid)

        val cids =
            if (inputData?.biddableConditionType == null)
                campaignsRepository.getCids(shard, lastCid, CAMPAIGNS_PER_ITERATION, BIDS_CHUNK)
            else
                bsExportBidRepositoryDispatcher.bidRepositories[inputData.biddableConditionType]!!
                    .getCids(shard, lastCid, BIDS_CHUNK)
        logger.info("got {} cids", cids.size)

        if (cids.isEmpty()) return null

        val count = bsExportBidRepositoryDispatcher.bidRepositories
            .filterKeys { inputData?.biddableConditionType == null || it == inputData.biddableConditionType }
            .values
            .map { process(shard, cids, it) }
            .sum()

        if (cids.last() >= maxCid)
            return null
        else
            return BiddableShowConditionsUploadToYtState(
                lastCampaignId = cids.last(),
                maxCampaignId = maxCid,
                importedCount = (prevState?.importedCount ?: 0) + count
            )
    }

    fun <T : BsExportAbstractBid> process(
        shard: Int,
        cids: List<Long>,
        repository: BsExportBidsRepositoryInterface<T>
    ): Long {
        var last: T? = null
        var count = 0L
        while (true) {
            val bids = repository.getBidsByCids(shard, cids, BIDS_CHUNK, last)
            if (bids.isEmpty()) break

            bsExportBiddableShowConditionsService.fillMissedOrderId(shard, bids)

            logger.info(
                "got {} rows of {}, shard: {}, from: {}, cids range: {} - {}",
                bids.size, bids[0]::class.simpleName, shard, last, cids.first(), cids.last()
            )

            val now = mapper.now()
            uploadYtRepository.modify(mapper.bidsToYtRecords(bids, now).map { it.bid })

            last = bids.last()
            count += bids.size
        }
        return count
    }
}
