package ru.yandex.direct.oneshot.oneshots.bsexport;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.Record2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.BsExportMultipliersObject;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.MultiplierType;
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.UpsertInfo;
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.BsExportMultipliersService;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.Retries;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;


@Component
@Multilaunch
@PausedStatusOnFail
@Retries(5)
@Approvers({"mspirit", "zakhar", "pema4"})
public class BsExportMultipliersOneshot
        implements ShardedOneshot<BsExportMultipliersOneshot.InputData, BsExportMultipliersOneshot.State> {
    private static final Logger logger = LoggerFactory.getLogger(BsExportMultipliersOneshot.class);

    private static final int CHUNK_SIZE = 5_000;
    private static final Map<HierarchicalMultipliersType, MultiplierType> TYPE_MAPPING =
            StreamEx.of(MultiplierType.values())
                    .mapToEntry(MultiplierType::getDbTypes, Function.identity())
                    .flatMapKeys(Collection::stream)
                    .toMap();
    private static final Set<String> SUPPORTED_TYPES = StreamEx.of(TYPE_MAPPING.keySet())
            .map(HierarchicalMultipliersType::getLiteral)
            .toSet();


    private final DslContextProvider dslContextProvider;
    private final BsExportMultipliersService bsExportMultipliersService;

    public BsExportMultipliersOneshot(DslContextProvider dslContextProvider,
                                      BsExportMultipliersService bsExportMultipliersService) {
        this.dslContextProvider = dslContextProvider;
        this.bsExportMultipliersService = bsExportMultipliersService;
    }

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        ItemValidationBuilder<InputData, Defect> vb = ItemValidationBuilder.of(inputData);
        vb.list(inputData.bidModifierTypes, "dbTypes")
                .check(notNull())
                .check(notEmptyCollection())
                .checkEach(inSet(SUPPORTED_TYPES));
        return vb.getResult();
    }

    @Nullable
    @Override
    public State execute(InputData inputData, State prevState, int shard) {
        if (prevState == null) {
            logger.info("First iteration! shard: {}", shard);
            return new State(0L);
        }

        var logicObjects = getBsExportMultipliersObjects(
                shard, prevState.lastHierarchicalMultiplierId, inputData.bidModifierTypes);

        if (logicObjects.isEmpty()) {
            logger.info("Last iteration, last hierarchical_multiplier_id: {}, shard: {}",
                    prevState.lastHierarchicalMultiplierId, shard);
            return null;
        }

        bsExportMultipliersService.updateMultipliers(shard, logicObjects);

        UpsertInfo lastUpsertInfo = logicObjects.get(logicObjects.size() - 1).getUpsertInfo();
        assert lastUpsertInfo != null;
        return new State(lastUpsertInfo.getHierarchicalMultiplierId());
    }

    private List<BsExportMultipliersObject> getBsExportMultipliersObjects(int shard, Long lastHierarchicalMultiplierId,
                                                                          List<String> bidModifierTypes) {
        return dslContextProvider.ppc(shard)
                .select(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID, HIERARCHICAL_MULTIPLIERS.TYPE)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(
                        HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.greaterThan(lastHierarchicalMultiplierId)
                                .and(HIERARCHICAL_MULTIPLIERS.TYPE.in(bidModifierTypes))
                )
                .orderBy(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID)
                .limit(CHUNK_SIZE)
                .fetch(this::convert);
    }

    @Nonnull
    private BsExportMultipliersObject convert(Record2<Long, HierarchicalMultipliersType> record) {
        MultiplierType multiplierType = checkNotNull(TYPE_MAPPING.get(record.value2()));
        return BsExportMultipliersObject.upsert(
                new UpsertInfo(multiplierType, record.value1()), 0L, "", "");
    }

    static class InputData {
        final List<String> bidModifierTypes;

        InputData(List<String> bidModifierTypes) {
            this.bidModifierTypes = bidModifierTypes;
        }
    }

    static class State {
        final Long lastHierarchicalMultiplierId;

        State(Long lastHierarchicalMultiplierId) {
            this.lastHierarchicalMultiplierId = lastHierarchicalMultiplierId;
        }
    }
}


