package ru.yandex.direct.oneshot.oneshots.bsexport

import org.jooq.Record1
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.BsExportMultipliersObject
import ru.yandex.direct.ess.logicobjects.bsexport.multipliers.TimeTargetChangedInfo
import ru.yandex.direct.logicprocessor.processors.bsexport.multipliers.BsExportMultipliersService
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class State(
    val lastCampaignId: Long
)

@Component
@Multilaunch
@PausedStatusOnFail
@Retries(5)
@Approvers("mspirit", "zakhar", "pema4")
class BsExportMultipliersTimeTargetOneshot @Autowired constructor(
    private val dsl: DslContextProvider,
    private val bsExportMultipliersService: BsExportMultipliersService
) : ShardedOneshot<Void, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(BsExportMultipliersTimeTargetOneshot::class.java)
        private const val CHUNK_SIZE = 5_000L
    }

    override fun validate(inputData: Void?): ValidationResult<Void, Defect<Any>> {
        return ValidationResult.success(inputData)
    }

    override fun execute(inputData: Void?, prevState: State?, shard: Int): State? {
        if (prevState == null) {
            logger.info("First iteration! shard: {}", shard)
            return State(0)
        }

        val logicObjects: List<BsExportMultipliersObject> = getCampaignsWithTimeTarget(shard, prevState.lastCampaignId)

        if (logicObjects.isEmpty()) {
            logger.info("Last iteration, last campaignId: {}, shard: {}", prevState.lastCampaignId, shard)
            return null
        }
        bsExportMultipliersService.updateMultipliers(shard, logicObjects)

        val lastUpsertInfo: TimeTargetChangedInfo = logicObjects[logicObjects.size - 1].timeTargetChangedInfo!!
        return State(lastUpsertInfo.campaignId)
    }

    private fun getCampaignsWithTimeTarget(shard: Int, lastCampaignId: Long): List<BsExportMultipliersObject> {
        return dsl.ppc(shard)
            .select(CAMPAIGNS.CID)
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.CID.greaterThan(lastCampaignId))
            .orderBy(CAMPAIGNS.CID.asc())
            .limit(CHUNK_SIZE)
            .fetch(this::convert)
    }

    private fun convert(record: Record1<Long>): BsExportMultipliersObject {
        val campaignId = record.value1()
        return BsExportMultipliersObject.timeTargetChanged(
            TimeTargetChangedInfo(campaignId), 0L, "", "")
    }
}
