package ru.yandex.direct.oneshot.oneshots.bsexport.altertable

import java.io.BufferedReader
import java.io.InputStreamReader
import java.net.URL
import java.time.Duration
import java.time.Instant
import java.util.UUID
import java.util.concurrent.CompletableFuture
import java.util.concurrent.TimeUnit
import com.google.gson.Gson
import com.google.gson.JsonArray
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.bolts.collection.impl.EmptyMap
import ru.yandex.direct.env.Environment
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.DefectId
import ru.yandex.direct.validation.util.check
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.inside.yt.kosher.common.GUID
import ru.yandex.inside.yt.kosher.cypress.YPath
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeBooleanNodeImpl
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeIntegerNodeImpl
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode
import ru.yandex.inside.yt.kosher.ytree.YTreeNode
import ru.yandex.yt.ytclient.proxy.YtClient
import ru.yandex.yt.ytclient.proxy.request.*
import ru.yandex.yt.ytclient.tables.ColumnSortOrder
import ru.yandex.yt.ytclient.tables.ColumnValueType


data class Param(
    val ytCluster: YtCluster,
    val tablePath: String,
    val schemaUrl: String,
    val incompleteSchema: Boolean,
    val replicatedTablets: Int?,
)

data class AlterTableState(
    val replicaSettings: Collection<ReplicaSettings>? = null,
    val pivotKeys: List<List<YTreeNode>>? = null,
)

data class ReplicaSettings(
    val path: String,
    val cluster: String,
    val mode: String,
    val state: String
)


/**
 * Ваншот для альтера таблиц на yt для нового транспорта в БК.
 * В качестве параметра принимает
 * а) ytCluster - кластер реприцируемой таблицы, или обычной, если она нереплицируемая
 * б) tablePath - таблица для альтера
 * в) schemaUrl - url новой схемы таблицы. Это схема отсюда https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs-internal/bstransport-yt/schemas,
 * загруженная, например, через ya upload в sandbox или s3. Например, https://proxy.sandbox.yandex-team.ru/1905432829
 * г) incompleteSchema - флаг, позволяющий указывать неполную схему, а она автоматически дополнится до текущей. будут добавлены только те колонки, которых нет в текущей схеме
 *
 * Если таблица нереплицированная, то выполняется обычный альтер
 *
 * Если таблица реплицированная, то выполняет альтер по немного измененному алгоритму https://yt.yandex-team.ru/docs/description/dynamic_tables/replicated_dynamic_tables#alter_schema
 * 0) проверяется, что все реплики доступны, иначе ваншот аварийно завершает работу
 * 1) замораживается реплицируемая таблица
 * 2) ожидается, когда все реплики получат все записанные строки
 * 3) пересоздается реплицируемая таблица
 * 4) создаются новые объекты реплики, аналогичные предыдущим
 * 5) выполняется alter для таблиц-реплик
 * 6) таблицам репликам выставляются новые upstream-replica-id
 * 7) реплики включаются
 *
 * Поддерживается работа с решардированными реплицированными таблицами:
 * * если параметр tablets не задан, то реплицируемая таблица шардируется так же, как и раньше
 * * если параметр tablets равен 1, реплицируемая таблица перестаёт шардироваться
 * * иначе реплицируемая таблица шардируется на заданное число таблетов
 */
@Component
@Approvers("mspirit", "eboguslavskaya", "zakhar", "ssdmitriev")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class AlterBsExportTableOneshot @Autowired constructor(private val ytProvider: YtProvider)
    : SimpleOneshot<Param, AlterTableState?> {
    companion object {
        private val logger = LoggerFactory.getLogger(AlterBsExportTableOneshot::class.java)
    }

    override fun validate(inputData: Param) = validateObject(inputData) {
        property(inputData::ytCluster) {
            check(CommonConstraints.notNull())
        }
        if (result.hasAnyErrors()) return@validateObject

        property(inputData::tablePath) {
            check(CommonConstraints.notNull())
            check(fromPredicate(
                { tableName -> ytProvider.getOperator(inputData.ytCluster).exists(YtTable(tableName)) },
                CommonDefects.objectNotFound()))
        }

        property(inputData::schemaUrl) {
            check(CommonConstraints.notNull())
            check(CommonDefects.objectNotFound()) { schemaUrl ->
                try {
                    URL(schemaUrl).content
                    true
                } catch (e: Exception) {
                    false
                }
            }
            check(ytHashModuloMustBeSame()) { schemaUrl ->
                try {
                    val ytClient = ytProvider.getDynamicOperator(inputData.ytCluster).ytClient
                    var newSchema = fetchColumnSchema(URL(schemaUrl))
                    if (inputData.incompleteSchema) {
                        newSchema = getFullSchema(ytClient, inputData.tablePath, newSchema)
                    }
                    val newYtHashModulo = getYtHashModuloOrNull(newSchema)
                    val oldSchema = ytProvider.getDynamicOperator(inputData.ytCluster).ytClient
                        .getNode("${inputData.tablePath}/@schema")
                        .getWithDefaultTimeout()
                        .listNode()
                    val oldYtHashModulo = getYtHashModuloOrNull(oldSchema)
                    newYtHashModulo == oldYtHashModulo
                } catch (e: Exception) {
                    false
                }
            }
        }

        property(inputData::incompleteSchema) {
            check(cantUseIncompleteSchemeInProduction()) { incompleteSchema ->
                !(incompleteSchema && Environment.getCached() == EnvironmentType.PRODUCTION)
            }
        }

        property(inputData::replicatedTablets) {
            if (inputData.replicatedTablets != null) {
                check(NumberConstraints.greaterThan(0))
            }
        }
    }

    override fun execute(inputData: Param, prevState: AlterTableState?): AlterTableState? {

        val state = prevState ?: AlterTableState()
        logger.info("Get table schema from ${inputData.schemaUrl}")
        val schemaUrl = URL(inputData.schemaUrl)
        var columnSchema = fetchColumnSchema(schemaUrl)
        val ytClient = ytProvider.getDynamicOperator(inputData.ytCluster).ytClient
        if (inputData.incompleteSchema) {
            columnSchema = getFullSchema(ytClient, inputData.tablePath, columnSchema)
        }
        return if (isReplicatedTable(ytClient, inputData.tablePath))
            alterReplicatedTable(ytClient, inputData, state, columnSchema)
        else {
            alterSimpleTable(ytClient, inputData.tablePath, columnSchema)
            null
        }
    }

    private fun isReplicatedTable(ytClient: YtClient, tablePath: String) =
        ytClient.getNode("$tablePath/@type").getWithDefaultTimeout()
            .stringValue() == "replicated_table"

    private fun alterReplicatedTable(
        ytClient: YtClient,
        inputParam: Param,
        state: AlterTableState,
        columnSchema: YTreeListNode,
    ): AlterTableState? {
        val tablePath = inputParam.tablePath
        logger.info("Table $tablePath is replicated, alter by instruction https://yt.yandex-team.ru/docs/description/dynamic_tables/replicated_dynamic_tables#alter_schema")
        val isTableExists = ytClient.existsNode(tablePath).getWithDefaultTimeout()

        // должна или существовать таблица, или сохраненная информация о репликах
        if (!isTableExists && state.replicaSettings == null)
            throw IllegalStateException("Table not exists and replicas settings not exists")

        // если настроек реплик нет в стейте, то сохраняем их туда, потому что после удаления таблицы мы уже не узнаем настройки реплик
        if (state.replicaSettings == null) {
            return AlterTableState(
                replicaSettings = getTableReplicas(ytClient, tablePath).values,
                pivotKeys = calculatePivotKeysForResharding(ytClient, tablePath, columnSchema, inputParam.replicatedTablets)
            )
        }

        // если кластеры реплик недоступны - лучше всего будет ничего не делать
        val inaccessibleReplicas = state.replicaSettings.filterNot(ytProvider::isReplicaAccessible)
        check(inaccessibleReplicas.isEmpty()) {
            "Replicas $inaccessibleReplicas are inaccessible, stopping iteration..."
        }

        if (isTableExists) {
            tryDoAlter(ytClient, tablePath, columnSchema)
            freezeTableAndRemove(ytClient, tablePath)
        }

        createReplicatedTable(tablePath, ytClient, columnSchema)
        if (state.pivotKeys != null) {
            logger.info("Resharding table, number of tablets: ${state.pivotKeys.size}")
            ytProvider[inputParam.ytCluster].tables()
                .reshard(YPath.simple(tablePath), state.pivotKeys)
        }

        state.replicaSettings
            .map { createReplica(ytClient, tablePath, it) to it }
            .forEach { (replicaId, replicaSettings) -> alterReplica(replicaId, replicaSettings, columnSchema) }

        logger.info("Mount replicated table $tablePath")
        ytClient.mountTable(tablePath).getWithDefaultTimeout()

        logger.info("Setting max_sync_replica_count=1 and enable_replicated_table_tracker attributed=true for replicated table $tablePath")
        ytClient.setNode("$tablePath/@replicated_table_options/max_sync_replica_count", YTreeIntegerNodeImpl(false, 1, EmptyMap()))
            .getWithDefaultTimeout()
        ytClient.setNode("$tablePath/@replicated_table_options/enable_replicated_table_tracker", YTreeBooleanNodeImpl(true, EmptyMap()))
            .getWithDefaultTimeout()
        if (state.pivotKeys != null) {
            logger.info("Setting tablet_balancer_config/enable_auto_reshard=false, because table was resharded ")
            ytClient.setNode(
                "$tablePath/@tablet_balancer_config/enable_auto_reshard",
                YTree.booleanNode(false)
            ).getWithDefaultTimeout()
        }

        logger.info("Successful")
        return null
    }

    private fun calculatePivotKeysForResharding(
        ytClient: YtClient,
        tablePath: String,
        columnSchema: YTreeListNode,
        replicatedTablets: Int?,
    ): List<List<YTreeNode>>? {
        val newTablets = when {
            replicatedTablets == 1 -> {
                logger.info("replicatedTablets=1, so table won't be resharded manually")
                null
            }
            replicatedTablets != null -> {
                logger.info("Replicated table will be resharded manually using $replicatedTablets tablets")
                replicatedTablets
            }
            !ytClient.isAutoResharded(tablePath) -> {
                val newTablets = ytClient.getTabletsNumber(tablePath)
                logger.info("Replicated table was resharded manually using $newTablets tablets, so it will be resharded again")
                newTablets
            }
            else -> {
                logger.info("Replicated table was not resharded before, so it won't be resharded now")
                null
            }
        }
        return newTablets?.let { generatePivotKeys(newTablets, columnSchema) }
    }

    /**
     * Замораживает таблицу на запись, дожидается, пока все реплики вычитают всё, удаляет таблицу
     */
    private fun freezeTableAndRemove(ytClient: YtClient, tablePath: String) {
        val replicaIdToSettings = getTableReplicas(ytClient, tablePath)
        logger.info("Freeze replicated table $tablePath")
        val tabletState = ytClient.getNode("$tablePath/@tablet_state").getWithDefaultTimeout()
            .stringValue()
        if (tabletState == "mounted") {
            ytClient.freezeTable(tablePath).getWithDefaultTimeout()
            waitState(ytClient, tablePath, "frozen")
        } else if (tabletState != "frozen" && tabletState != "unmounted") {
            throw IllegalStateException("Table $tablePath should be frozen or unmounted before alter")
        }


        logger.info("Got replicas with settings: $replicaIdToSettings")
        try {
            replicaIdToSettings.keys
                .forEach { waitReplicaFinishesReplication(it, ytClient) }
        } catch (ex: Exception) {
            ytClient.unfreezeTable(tablePath).getWithDefaultTimeout()
            throw ex
        }
        logger.info("All replicas finished replication, let's do alter")

        logger.info("Unmount replicated table $tablePath")
        unmountTableWithWait(tablePath, ytClient)

        logger.info("Remove replicated table $tablePath")
        ytClient.removeNode(tablePath).getWithDefaultTimeout()
    }

    private fun getTableReplicas(ytClient: YtClient, tablePath: String): Map<String, ReplicaSettings> {
        val replicas = ytClient.getNode("$tablePath/@replicas").getWithDefaultTimeout()
        return replicas.asMap()
            .map { (replicaId, replicaSettingsNode) -> replicaId to getReplicaSettings(replicaSettingsNode.mapNode()) }
            .toMap()
    }

    private fun waitReplicaFinishesReplication(replicaId: String, ytClient: YtClient) {
        logger.info("Wait for all tablets of replica $replicaId will finish replication (flushed_row_count = current_replication_row_index)")
        val waitStartTime = Instant.now()
        while (true) {
            val replicaTablets = ytClient.getNode("#$replicaId/@tablets").getWithDefaultTimeout()
            val unFinishedTablets = replicaTablets.asList()
                .map { it.asMap() }
                .filterNot { it.get("flushed_row_count")?.intValue() == it.get("current_replication_row_index")?.intValue() }
            when {
                unFinishedTablets.isEmpty() -> {
                    return
                }
                Duration.between(waitStartTime, Instant.now()) > Duration.ofMinutes(1) -> {
                    throw IllegalStateException(
                        "Replica $replicaId hadn't finished replication in 1 minute, terminating..."
                    )
                }
                else -> {
                    logger.info("Waiting for 1 sec, some tablets haven't finished replication: $unFinishedTablets")
                    Thread.sleep(1000L)
                }
            }
        }

    }

    private fun createReplica(ytClient: YtClient, tablePath: String, replicaSettings: ReplicaSettings): String {
        logger.info("Create table replica for table $tablePath with settings $replicaSettings")
        val replicaAttributes = YTree.attributesBuilder()
            .key("table_path").value(tablePath)
            .key("cluster_name").value(replicaSettings.cluster)
            .key("replica_path").value(replicaSettings.path)
            .key("mode").value(replicaSettings.mode)
            .buildAttributes()

        val createReplicaRequest = CreateObject(ObjectType.TableReplica).setAttributes(replicaAttributes)
        val guid = ytClient.createObject(createReplicaRequest).getWithDefaultTimeout()

        logger.info("Created table replica with guid = $guid")
        logger.info("Enable table replica $guid")
        if (replicaSettings.state == "enabled") {
            ytClient.alterTableReplica(AlterTableReplica(guid).setEnabled(true)).getWithDefaultTimeout()
        }
        return guid.toString()
    }

    private fun alterReplica(replicaId: String, replicaSettings: ReplicaSettings, columnSchema: YTreeNode) {
        val ytClient = ytProvider.getDynamicOperator(YtCluster.parse(replicaSettings.cluster)).ytClient
        alterSimpleTable(ytClient, replicaSettings.path, columnSchema, replicaId)
    }

    private fun alterSimpleTable(ytClient: YtClient, tablePath: String, columnSchema: YTreeNode, upstreamReplicaId: String? = null) {
        logger.info("Alter simple table $tablePath on cluster $ytClient with upstream replica id = $upstreamReplicaId")
        logger.info("Unmount table $tablePath")
        unmountTableWithWait(tablePath, ytClient)

        val alterRequest = AlterTable(tablePath).setSchema(columnSchema)
        upstreamReplicaId?.let { alterRequest.setUpstreamReplicaId(GUID.valueOf(upstreamReplicaId)) }

        logger.info("Alter table $tablePath, set schema = $columnSchema")
        ytClient.alterTable(alterRequest).getWithDefaultTimeout()

        logger.info("Mount table $tablePath")
        ytClient.mountTable(tablePath).getWithDefaultTimeout()
    }

    /**
     * Создает реплицируемую таблицу с указанной схемой
     */
    private fun createReplicatedTable(tablePath: String, ytClient: YtClient, columnSchema: YTreeListNode): GUID {
        logger.info("Create replicated table $tablePath with new schema")
        val createSchema = YTree.listBuilder().buildList()
        createSchema.addAll(columnSchema.asList())
        createSchema.putAttribute("unique_keys", YTreeBooleanNodeImpl(true, EmptyMap()))
        createSchema.putAttribute("strict", YTreeBooleanNodeImpl(true, EmptyMap()))

        val tableAttributes = YTree.mapBuilder()
            .key("dynamic").value(true)
            .key("schema").value(createSchema)
            .buildMap()
            .asMap()

        logger.info("New table schema $tableAttributes")
        return ytClient
            .createNode(CreateNode(tablePath, ObjectType.ReplicatedTable, tableAttributes))
            .getWithDefaultTimeout()
    }

    private fun unmountTableWithWait(tablePath: String, ytClient: YtClient) {
        ytClient.unmountTable(tablePath).getWithDefaultTimeout()
        do {
            val mountedTablets = ytClient.getNode("$tablePath/@tablets").getWithDefaultTimeout()
                .listNode()
                .filter { it.mapNode().get("state").get().stringValue() != "unmounted" }
            if (mountedTablets.isNotEmpty()) {
                logger.info("Not all tablets of table $tablePath unmounted, waiting for 1 sec")
                Thread.sleep(1000L)
            }
        } while (mountedTablets.isNotEmpty())
        return
    }

    private fun waitState(ytClient: YtClient, tablePath: String, state: String) {
        do {
            val tabletState = ytClient.getNode("$tablePath/@tablet_state").getWithDefaultTimeout()
                .stringValue()
            if (tabletState != state) {
                logger.info("Table $tablePath state $tabletState, expected $state")
                Thread.sleep(1000L)
            }
        } while (tabletState != state)
    }

    private fun getReplicaSettings(replicaSettingsNode: YTreeMapNode): ReplicaSettings =
        ReplicaSettings(
            path = replicaSettingsNode.mapNode().getString("replica_path"),
            cluster = replicaSettingsNode.mapNode().getString("cluster_name"),
            mode = replicaSettingsNode.mapNode().getString("mode"),
            state = replicaSettingsNode.mapNode().getString("state"))

    private fun fetchColumnSchema(schemaUrl: URL): YTreeListNode {
        val reader = BufferedReader(
            InputStreamReader(schemaUrl.openStream()))

        val gson = Gson()
        val schemaArray = gson.fromJson(reader, JsonArray::class.java)

        return getColumnSchema(schemaArray)
    }

    /**
     * Преобразует схемы столбцов из json в YTree
     */
    private fun getColumnSchema(jsonArraySchema: JsonArray): YTreeListNode {
        val columnSchemaBuilder = YTree.listBuilder()
        jsonArraySchema.map {
            val name = it.asJsonObject.get("name").asString
            val type = ColumnValueType.fromName(it.asJsonObject.get("type").asString)
            val required = it.asJsonObject.get("type")?.asBoolean ?: false
            val sortOrder = it.asJsonObject.get("sort_order")?.asString?.let { sortOrder -> ColumnSortOrder.fromName(sortOrder) }
            val expression = it.asJsonObject.get("expression")?.asString
            val group = it.asJsonObject.get("group")?.asString

            columnSchemaBuilder
                .beginMap()
                .key("name").value(name)
                .key("type").value(type.getName())
                .key("required").value(required)
            if (sortOrder != null) {
                columnSchemaBuilder.key("sort_order").value(sortOrder.getName())
            }
            if (expression != null) {
                columnSchemaBuilder.key("expression").value(expression)
            }
            if (group != null) {
                columnSchemaBuilder.key("group").value(group)
            }
            columnSchemaBuilder.endMap()
        }
        return columnSchemaBuilder.buildList()
    }

    /**
     * Пробует сделать альтер на временной таблице со схемой, как у исходной таблицы
     * Нужно, чтобы проверить, новая схема применима к старой, до удаления реплицируемой таблицы
     */
    private fun tryDoAlter(ytClient: YtClient, tablePath: String, newColumnSchema: YTreeListNode) {
        val tmpTablePath = "//tmp/adv/${UUID.randomUUID()}"
        val currentSchema = ytClient.getNode("$tablePath/@schema").getWithDefaultTimeout()

        val tableAttributes = YTree.mapBuilder()
            .key("dynamic").value(true)
            .key("schema").value(currentSchema)
            .buildMap()
            .asMap()

        logger.info("New table schema $tableAttributes")
        ytClient
            .createNode(CreateNode(tmpTablePath, ObjectType.Table, tableAttributes).setRecursive(true))
            .getWithDefaultTimeout()
        try {
            ytClient.alterTable(AlterTable(tmpTablePath).setSchema(newColumnSchema))
                .getWithDefaultTimeout()
        } catch (e: RuntimeException) {
            logger.info("Failed to alter tmp table: $e")
            throw e
        }
        logger.info("Successfully alter temp table")
        ytClient.removeNode(tmpTablePath).getWithDefaultTimeout()
    }

    private fun getFullSchema(ytClient: YtClient, tablePath: String, newSchema: YTreeListNode): YTreeListNode {
        val currentSchema = ytClient.getNode("$tablePath/@schema").getWithDefaultTimeout()
        val currentSchemaColumnNames = currentSchema.listNode()
            .map { it.mapNode().get("name").get().stringValue() }
            .toSet()
        val resultSchemaBuilder = YTree.listBuilder()
        currentSchema.listNode()
            .forEach { resultSchemaBuilder.value(it) }

        newSchema.listNode()
            .filterNot { currentSchemaColumnNames.contains(it.mapNode().get("name").get().stringValue()) }
            .forEach { resultSchemaBuilder.value(it) }
        return resultSchemaBuilder.buildList()

    }

    private fun generatePivotKeys(tablets: Int, schema: YTreeListNode): List<List<YTreeNode>>? {
        // взято из https://a.yandex-team.ru/arc_vcs/direct/libs-internal/bstransport-yt/scripts/reshard_table.py?rev=fd8aac7058c3c14084b6ebbb702f943546e5c174#L15
        val modulo = getYtHashModuloOrNull(schema) ?: return null
        val pivotKeys = List(tablets) { idx ->
            when (idx) {
                0 -> listOf()
                else -> listOf(YTree.unsignedLongNode(idx * modulo / tablets))
            }
        }
        check(pivotKeys.distinct().size == tablets) {
            "Pivot keys are not distinct, consider using less tablets"
        }
        return pivotKeys
    }

    private fun getYtHashModuloOrNull(schema: YTreeListNode): Long? {
        val ytHashColumn = schema.listNode()
            .find { it.mapNode().getOrThrow("name").stringValue() == "YtHash" } ?: return null

        val ytHashExpression = ytHashColumn
            .mapNode()
            .getStringO("expression")
            .orElse(null)
            .replace(" ", "")

        return "farm_hash\\(.*\\)%(\\d+)".toRegex(RegexOption.IGNORE_CASE)
            .find(ytHashExpression)
            ?.groupValues
            ?.get(1)
            ?.toLong()
    }

    private fun YtClient.isAutoResharded(tablePath: String): Boolean =
        getNode("$tablePath/@tablet_balancer_config/enable_auto_reshard")
            .getWithDefaultTimeout()
            .boolValue()

    private fun YtClient.getTabletsNumber(tablePath: String): Int =
        getNode("$tablePath/@resource_usage/tablet_count")
            .getWithDefaultTimeout()
            .intValue()

    private fun <T> CompletableFuture<T>.getWithDefaultTimeout(): T {
        return get(60, TimeUnit.SECONDS)
    }
}

private fun YtProvider.isReplicaAccessible(replicaSettings: ReplicaSettings): Boolean = runCatching {
    val cluster = YtCluster.clusterFromNameToUpperCase(replicaSettings.cluster)
    val operator = getOperator(cluster)
    val requestedTable = YtTable(replicaSettings.path)
    operator.exists(requestedTable)
}.getOrDefault(false)

private enum class AlterTable : DefectId<Void> {
    CANT_USE_INCOMPLETE_SCHEME_IN_PRODUCTION_ENV,
    YT_HASH_MODULO_MUST_BE_SAME,
}

private fun cantUseIncompleteSchemeInProduction(): Defect<Void> {
    return Defect(AlterTable.CANT_USE_INCOMPLETE_SCHEME_IN_PRODUCTION_ENV)
}

private fun ytHashModuloMustBeSame() =
    Defect(AlterTable.YT_HASH_MODULO_MUST_BE_SAME)
