package ru.yandex.direct.oneshot.oneshots.bsexport.autobudget.restart

import org.jooq.impl.DSL
import org.slf4j.LoggerFactory.getLogger
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartData
import ru.yandex.direct.autobudget.restart.model.StrategyDto
import ru.yandex.direct.autobudget.restart.repository.CampaignAutobudgetRestartRepository
import ru.yandex.direct.autobudget.restart.repository.CampRestartData
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.repository.SaveMode
import ru.yandex.direct.autobudget.restart.service.Reason
import ru.yandex.direct.autobudget.restart.service.StrategyState
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.hasEnableCpcHoldFromDb
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_METRIKA_GOALS
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS
import ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.SqlUtils
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.utils.JsonUtils.fromJson
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import java.time.LocalDate
import java.time.LocalDateTime

data class Param(
    val orderIdPercent: Long,
    val mode: SaveMode = SaveMode.IGNORE_EXISTING
)

data class State(
    val lastCid: Long
)

@Component
@Approvers("zhur", "hmepas", "lena-san")
@Multilaunch
@PausedStatusOnFail
class AutobudgetRestartInitOneshot @Autowired constructor(
    private val dsl: DslContextProvider,
    private val repo: CampaignAutobudgetRestartRepository
) : ShardedOneshot<Param, State?> {
    companion object {
        private val logger = getLogger(AutobudgetRestartInitOneshot::class.java)
        private const val LIMIT = 5_000L

        private val INIT_RESTART_TIME = LocalDateTime.parse("2020-01-01T00:00:00")
        private val OLD_START_TIME = LocalDate.parse("2000-01-01")
        private val OLD_STOP_TIME = LocalDateTime.parse("2020-01-01T00:00:00")

        // запрос совпадает с $SQL_HAS_MONEY в BS::ExportWorker
        private val HAS_MONEY_FIELD =
            DSL.field(
                "c.sum + IFNULL(wc.sum, 0) + clo.auto_overdraft_lim" +
                        " - c.sum_spent - IFNULL(wc.sum_spent, 0)" +
                        " > ?", Boolean::class.java, Currencies.EPSILON
            )
                .`as`("has_money")

    }

    override fun validate(inputData: Param) =
        validateObject(inputData) {
            property(inputData::orderIdPercent) {
                check(CommonConstraints.notNull())
                check(NumberConstraints.notLessThan(1L))
                check(NumberConstraints.notGreaterThan(100L))
            }
        }

    override fun execute(inputData: Param, prevState: State?, shard: Int): State? {
        val lastCid = prevState?.lastCid ?: 0L
        logger.info("Start from cid=$lastCid, shard=$shard")

        val strategyData = getCampStrategyData(shard, lastCid, inputData.orderIdPercent)
        val newLastCid = strategyData.map { it.cid }.maxOrNull()
        logger.info("Selected ${strategyData.size} rows, shard=$shard")
        if (newLastCid == null) return null

        val restartData = calculateInitState(strategyData)
        repo.saveAutobudgetRestartData(shard, restartData, inputData.mode)

        if (strategyData.size < LIMIT) {
            logger.info("There are no rows for changing, shard {}", shard)
            return null
        }

        logger.info("End of iteration, new lastCid=$newLastCid, shard=$shard")
        return State(newLastCid)
    }

    private fun calculateInitState(strategyData: List<CampStrategyRestartData>) =
        strategyData.map {
            CampRestartData(
                cid = it.cid,
                times = RestartTimes(
                    restartTime = INIT_RESTART_TIME,
                    softRestartTime = INIT_RESTART_TIME,
                    restartReason = Reason.INIT.name
                ),
                strategyData = it.strategyDto,
                state = StrategyState(
                    stopTime =
                    if (it.strategyDto.statusShow) null
                    else OLD_STOP_TIME
                )
            )
        }

    private fun getCampStrategyData(shard: Int, lastCid: Long, orderIdPercent: Long): List<CampStrategyRestartData> {
        val c = CAMPAIGNS.`as`("c")
        val co = CAMP_OPTIONS.`as`("co")
        val wc = CAMPAIGNS.`as`("wc")
        val clo = CLIENTS_OPTIONS.`as`("clo")

        val camps = dsl.ppc(shard)
            .select(
                c.CID, c.TYPE,
                c.STRATEGY_DATA,
                co.STRATEGY, c.PLATFORM,
                c.START_TIME, c.DAY_BUDGET, c.OPTS, c.TIME_TARGET, c.STATUS_SHOW,
                HAS_MONEY_FIELD
            )
            .from(c)
            .innerJoin(co).on(co.CID.eq(c.CID))
            .leftJoin(wc).on(wc.CID.eq(c.WALLET_CID))
            .leftJoin(clo).on(clo.CLIENT_ID.eq(c.CLIENT_ID))
            .where(
                c.CID.gt(lastCid)
                    .and(c.STRATEGY_DATA.isNotNull)
                    .and(c.TYPE.ne(CampaignsType.wallet))
                    .and(c.ORDER_ID.greaterThan(0))
                    .and(c.ORDER_ID.mod(100).lessThan(orderIdPercent))
            )
            .orderBy(c.CID)
            .limit(LIMIT)
            .fetch()

        if (camps.isEmpty()) return listOf()
        val cids = camps.map { it.into(c.recordType).cid }

        val g = CAMP_METRIKA_GOALS.`as`("g")
        val campsWithCombinedGoals = dsl.ppc(shard)
            .selectDistinct(g.CID)
            .from(g)
            .where(
                g.CID.`in`(cids)
                    .and(g.LINKS_COUNT.gt(0))
                    .and(SqlUtils.findInSet("combined", g.GOAL_ROLE).gt(0))
            )
            .fetchSet(g.CID)

        return camps.map {
            val campaign = it.into(c.recordType)
            val campOptions = it.into(co.recordType)
            val strategy = fromJson(campaign.strategyData, StrategyData::class.java)
            CampStrategyRestartData(
                campaign.cid,
                StrategyDto(
                    strategy = strategy.name,
                    manualStrategy = campOptions.strategy?.literal,
                    platform = campaign.platform?.literal,
                    startTime = campaign.startTime ?: OLD_START_TIME,
                    dayBudget = campaign.dayBudget,
                    autoBudgetSum = strategy.sum,
                    enableCpcHold = hasEnableCpcHoldFromDb(campaign.opts),
                    timeTarget = campaign.timetarget,
                    statusShow = campaign.statusshow == CampaignsStatusshow.Yes,
                    payForConversion = strategy.payForConversion ?: false,
                    goalId = strategy.goalId,
                    roiCoef = strategy.roiCoef,
                    limitClicks = strategy.limitClicks,
                    avgCpm = strategy.avgCpm,
                    avgBid = strategy.avgBid,
                    avgCpa = strategy.avgCpa,
                    hasCombinedGoals = campsWithCombinedGoals.contains(campaign.cid),
                    hasMoney = it.get(HAS_MONEY_FIELD),
                )
            )
        }
    }
}
