package ru.yandex.direct.oneshot.oneshots.bsexport.autobudget.restart

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.autobudget.restart.repository.CampaignAutobudgetRestartRepository
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import java.math.BigDecimal

data class FixAutoBudgetRestartForWalletsParam(
    val ytCluster: YtCluster,
    val tablePath: String
)

data class FixAutoBudgetRestartForWalletsState(
    val lastRow: Long
)

class FixAutoBudgetRestartForWalletsRow : YtTableRow(listOf(CID)) {
    companion object {
        private val CID = YtField("cid", Long::class.java)
    }

    val cid: Long?
        get() = valueOf(CID)
}

data class CampaignCurrentState(
    val statusBsSynced: CampaignsStatusbssynced,
    val dayBudget: BigDecimal,
    val platform: CampaignsPlatform?,
    val type: CampaignsType
)

@Component
@Approvers("mspirit", "elwood", "ppalex")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class FixAutoBudgetRestartForWalletsOneshot(
    private val dslContextProvider: DslContextProvider,
    private val ytProvider: YtProvider,
    private val shardHelper: ShardHelper,
    private val autobudgetRestartRepository: CampaignAutobudgetRestartRepository
) : SimpleOneshot<FixAutoBudgetRestartForWalletsParam, FixAutoBudgetRestartForWalletsState?> {
    companion object {
        private val logger = LoggerFactory.getLogger(FixAutoBudgetRestartForWalletsOneshot::class.java)
    }

    override fun validate(inputData: FixAutoBudgetRestartForWalletsParam) =
        validateObject(inputData) {
            property(inputData::tablePath) {
                check(CommonConstraints.notNull())
                check(
                    Constraint.fromPredicate(
                        { ytProvider.getOperator(inputData.ytCluster).exists(YtTable(inputData.tablePath)) },
                        CommonDefects.objectNotFound()
                    )
                )
            }
        }

    override fun execute(
        inputData: FixAutoBudgetRestartForWalletsParam,
        prevState: FixAutoBudgetRestartForWalletsState?
    ): FixAutoBudgetRestartForWalletsState? {
        logger.info("Start from state=$prevState")
        val startRow = prevState?.lastRow ?: 0
        val chunkSize = 100
        val lastRow = startRow + chunkSize
        val cids = getRowsFromYtTable(
            inputData.ytCluster,
            inputData.tablePath,
            startRow,
            lastRow
        )
        shardHelper.groupByShard(cids, ShardKey.CID)
            .forEach { shard, shardedCampaignIds ->
                shardedCampaignIds.forEach {
                    try {
                        dslContextProvider.ppcTransaction(shard) { conf ->
                            val dsl = conf.dsl()
                            val state = dsl.select(
                                CAMPAIGNS.STATUS_BS_SYNCED,
                                CAMPAIGNS.DAY_BUDGET,
                                CAMPAIGNS.PLATFORM,
                                CAMPAIGNS.TYPE
                            )
                                .from(CAMPAIGNS)
                                .where(CAMPAIGNS.CID.eq(it))
                                .forUpdate()
                                .fetchOne { r ->
                                    CampaignCurrentState(
                                        r.get(CAMPAIGNS.STATUS_BS_SYNCED),
                                        r.get(CAMPAIGNS.DAY_BUDGET),
                                        r.get(CAMPAIGNS.PLATFORM),
                                        r.get(CAMPAIGNS.TYPE),
                                    )
                                }
                            if (state.type != CampaignsType.wallet) {
                                logger.info("SKIP: Campaign $it is not wallet, skip it")
                                return@ppcTransaction
                            }
                            // Отправляем только синкнутые кампании, на остальных есть риск гонки, если кампания начнет обрабатываться транспортом в бк
                            // В этом случае можем просыпать рестарт автобюджета
                            if (state.statusBsSynced != CampaignsStatusbssynced.Yes) {
                                logger.info("SKIP: Campaign $it has status $state.statusBsSynced, skip it")
                                return@ppcTransaction
                            }

                            val restartDataList =
                                autobudgetRestartRepository.getAutobudgetRestartData(dsl, listOf(it), true)
                            if (restartDataList.isEmpty()) {
                                logger.info("SKIP: No restart data for campaign $it")
                                return@ppcTransaction
                            }
                            val strategyDto = restartDataList[0].strategyData
                            if (strategyDto.dayBudget != null) {
                                logger.info("SKIP: Day budget is not null")
                                return@ppcTransaction
                            }
                            val newStrategyDto = strategyDto.copy(
                                dayBudget = state.dayBudget,
                                platform = state.platform?.literal
                            )
                            val newRestartData = restartDataList[0].copy(
                                strategyData = newStrategyDto
                            )
                            autobudgetRestartRepository.saveAutobudgetRestartData(dsl, listOf(newRestartData))
                        }
                    } catch (e: Exception) {
                        logger.error("SKIP: Handle campaign $it failed with exception", e)
                    }
                }

            }
        if (cids.size < chunkSize) {
            return null
        }
        return FixAutoBudgetRestartForWalletsState(lastRow)
    }

    private fun getRowsFromYtTable(
        ytCluster: YtCluster,
        tablePath: String,
        startRow: Long,
        lastRow: Long
    ): List<Long> {
        val result = mutableListOf<Long>()
        ytProvider.getOperator(ytCluster)
            .readTableByRowRange(
                YtTable(tablePath),
                { result.add(it.cid!!) },
                FixAutoBudgetRestartForWalletsRow(),
                startRow,
                lastRow
            )
        return result
    }
}
