package ru.yandex.direct.oneshot.oneshots.bsexport.bids

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.RESYNC_BIDDABLE_SHOW_CONDITIONS_CHUNK_SIZE
import ru.yandex.direct.common.db.PpcPropertyNames.RESYNC_BIDDABLE_SHOW_CONDITIONS_RELAX_TIME
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.config.bsexport.bids.BsExportBiddableShowConditionsConfig
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BidObjectType
import ru.yandex.direct.ess.logicobjects.bsexport.bids.BsExportBidsObject
import ru.yandex.direct.oneshot.oneshots.bsexport.BaseResyncOneshot
import ru.yandex.direct.oneshot.oneshots.bsexport.ResyncTableRow
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtField

class BidResyncTableRow : ResyncTableRow<BsExportBidsObject>(
    listOf(CID, PID, ID, RESOURCE_TYPE, IS_DELETED)
) {
    val cid: Long
        get() = valueOf(CID)

    val pid: Long
        get() = valueOf(PID)

    val id: Long
        get() = valueOf(ID)

    val bidType: BidObjectType
        get() = BidObjectType.valueOf(valueOf(RESOURCE_TYPE))

    val isDeleted: Boolean?
        get() = valueOf(IS_DELETED)

    companion object {
        private val CID = YtField("cid", Long::class.java)
        private val PID = YtField("pid", Long::class.java)
        private val ID = YtField("id", Long::class.java)
        private val RESOURCE_TYPE = YtField("resource_type", String::class.java)
        private val IS_DELETED = YtField("is_deleted", Boolean::class.javaObjectType)
    }

    override fun convert() = BsExportBidsObject(cid, pid, id, bidType, isDeleted ?: false)
}

/**
 * Ваншот для переотправки выбранных групп в `//home/adv/DirectAdGroups` через ESS
 * Читает строки пачками, размер пачки на данный момнт регулируется пропертей RESYNC_AD_GROUPS_CHUNK_SIZE
 * Если пачка для обработки непустая, то после итерации делается пауза, которая регулируется пропертей RESYNC_AD_GROUPS_RELAX_TIME
 * @see BaseResyncOneshot
 */
@Component
@Approvers("mspirit", "zakhar", "pema4")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class BiddableShowConditionsResyncOneshot @Autowired constructor(
    ytProvider: YtProvider,
    essClient: EssClient,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val shardHelper: ShardHelper,
) : BaseResyncOneshot<BsExportBidsObject, BidResyncTableRow>(ytProvider, essClient) {
    override val chunkSizeProperty =
        ppcPropertiesSupport.get(RESYNC_BIDDABLE_SHOW_CONDITIONS_CHUNK_SIZE)

    override val relaxTimeProperty =
        ppcPropertiesSupport.get(RESYNC_BIDDABLE_SHOW_CONDITIONS_RELAX_TIME)

    override val essLogicProcessName = BsExportBiddableShowConditionsConfig().logicProcessName

    override fun createEmptyYtRow() = BidResyncTableRow()

    override fun getLogicObjects(shard: Int, requests: List<BsExportBidsObject>): List<BsExportBidsObject> {
        // отфильтровываем чужие шарды
        val allCids = requests.map { it.cid }.distinct()
        val shardCids = shardHelper.groupByShard(allCids, ShardKey.CID)
            .shardedDataMap
            .filter { it.key == shard }
            .flatMap { it.value }
            .toSet()
        return requests.filter { it.cid in shardCids }
    }
}
