package ru.yandex.direct.oneshot.oneshots.bsexport.fixbigbids

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.currency.Money
import ru.yandex.direct.dbschema.ppc.Tables.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

@Component
@Approvers("zhur", "mspirit", "lena-san")
@Multilaunch
@PausedStatusOnFail
class BsExportFixBigAutobudgetRetargetingBids @Autowired constructor(
    private val dsl: DslContextProvider,
    private val shardHelper: ShardHelper,
) : SimpleOneshot<Void, Void> {
    companion object {
        private val logger = LoggerFactory.getLogger(BsExportFixBigAutobudgetRetargetingBids::class.java)

        // большие ставки не пролезают в CaeSaR
        private val BIG_BID = Int.MAX_VALUE / 10_000

        // мера предосторожности - мы не хотим проаптетить по ошибке много записей
        private val LIMIT = 100
    }

    override fun validate(inputData: Void?) = ValidationResult.success<Void, Defect<Any>>(null)

    override fun execute(inputData: Void?, prevState: Void?): Void? {
        var updated = 0

        for (shard in shardHelper.dbShards()) {
            logger.info("start process shard {}", shard)

            val c = CAMPAIGNS.`as`("c")
            val br = BIDS_RETARGETING.`as`("br")

            val rows = dsl.ppc(shard).select(br.RET_ID, c.CURRENCY, br.PRICE_CONTEXT)
                .from(br)
                .join(c).on(c.CID.eq(br.CID))
                .where(br.PRICE_CONTEXT.gt(BIG_BID.toBigDecimal()))
                .limit(LIMIT)
                .fetch {
                    PriceData(
                        it[br.RET_ID],
                        Money.valueOf(it[br.PRICE_CONTEXT], CurrencyCode.valueOf(it[c.CURRENCY].literal))
                    )
                }

            logger.info("fetched {} rows", rows.size)
            assert(rows.size >= LIMIT) { "Too much bad bids_retargeting rows!" }

            for (row in rows) {
                val maxPrice = row.price.currencyCode.currency.maxPrice
                assert(row.price.bigDecimalValue() >= maxPrice) {
                    "Bid is big, but less than maxPrice"
                }
                logger.info("Fix bid for $row")
                updated += dsl.ppc(shard).update(BIDS_RETARGETING)
                    .set(BIDS_RETARGETING.PRICE_CONTEXT, maxPrice)
                    .where(BIDS_RETARGETING.RET_ID.eq(row.retId))
                    .and(BIDS_RETARGETING.PRICE_CONTEXT.ge(maxPrice))
                    .execute()
            }
        }

        logger.info("Totally updated {} rows", updated)

        return null;
    }

    data class PriceData(
        val retId: Long,
        val price: Money
    )
}
