package ru.yandex.direct.oneshot.oneshots.bsexport.resources

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.common.models.BaseLogicObject
import ru.yandex.direct.ess.config.bsexport.resources.BsExportBannerResourcesConfig
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BannerResourceType
import ru.yandex.direct.ess.logicobjects.bsexport.resources.BsExportBannerResourcesObject
import ru.yandex.direct.oneshot.oneshots.bsexport.BaseResyncOneshot
import ru.yandex.direct.oneshot.oneshots.bsexport.ResyncTableRow
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtField

data class BidWithResourceType(
    val bid: Long,
    val resourceType: BannerResourceType
)

class BannerResourceResyncTableRow : ResyncTableRow<BidWithResourceType>(listOf(BID, RESOURCE_TYPE)) {
    val bid: Long?
        get() = valueOf(BID)
    val resourceType: String?
        get() = valueOf(RESOURCE_TYPE)

    companion object {
        private val BID = YtField("bid", Long::class.java)
        private val RESOURCE_TYPE = YtField("resource_type", String::class.java)
        private val logger = LoggerFactory.getLogger(BannerResourcesResyncOneshot::class.java)
    }

    override fun convert(): BidWithResourceType {
        val resourceType: BannerResourceType
        try {
            resourceType = BannerResourceType.valueOf(this.resourceType ?: "")
        } catch (e: IllegalStateException) {
            logger.error("Unknown banner type={} for cid={}, skipping", this.resourceType, this.bid)
            throw e
        }
        return BidWithResourceType(
            bid ?: throw IllegalStateException("ERROR: found null cid"),
            resourceType
        )
    }

    val id: Long
        get() = bid ?: throw IllegalStateException("ERROR: found null cid")
}

/**
 * Ваншот для отправки баннеров в бк транспортом ESS
 * Читает строки пачками, размер пачки на данный момнт регулируется пропертей RESYNC_BANNER_RESOURCES_CHUNK_SIZE
 * Если пачка для обработки непустая, то после итерации делается пауза, которая регулируется пропертей RESYNC_BANNER_RESOURCES_RELAX_TIME
 * @see BaseResyncOneshot
 */
@Component
@Approvers("mspirit", "zakhar", "pema4")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class BannerResourcesResyncOneshot @Autowired constructor(
    ytProvider: YtProvider,
    essClient: EssClient,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val bannerTypedRepository: BannerTypedRepository,
) : BaseResyncOneshot<BidWithResourceType, BannerResourceResyncTableRow>(
    ytProvider, essClient
) {

    override val chunkSizeProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.RESYNC_BANNER_RESOURCES_CHUNK_SIZE)

    override val relaxTimeProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.RESYNC_BANNER_RESOURCES_RELAX_TIME)

    override val essLogicProcessName: String =
        BsExportBannerResourcesConfig.LOGIC_PROCESS_NAME

    override fun createEmptyYtRow(): BannerResourceResyncTableRow =
        BannerResourceResyncTableRow()

    override fun getLogicObjects(shard: Int, requests: List<BidWithResourceType>): List<BaseLogicObject> {
        val ids = requests.map { it.bid }.distinct()
        val entities = getEntitiesFromRepository(ids, shard)
            .associateBy { it.id }

        return requests
            .filter { it.bid in entities }
            .map { createLogicObject(it, entities.getValue(it.bid)) }
    }


    fun getEntitiesFromRepository(ids: Collection<Long>, shard: Int): List<BannerWithSystemFields> =
        bannerTypedRepository
            .getSafely(shard, ids, BannerWithSystemFields::class.java)

    fun createLogicObject(request: BidWithResourceType, entity: BannerWithSystemFields): BaseLogicObject =
        BsExportBannerResourcesObject.Builder()
            .setBid(entity.id)
            .setPid(entity.adGroupId)
            .setCid(entity.campaignId)
            .setBannerId(entity.bsBannerId)
            .setResourceType(request.resourceType)
            .setReqid(Trace.current().spanId)
            .setMethod(Trace.current().method)
            .setService(Trace.current().service)
            .build()
}
