package ru.yandex.direct.oneshot.oneshots.bsexport.strategies

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.ess.client.EssClient
import ru.yandex.direct.ess.common.models.BaseLogicObject
import ru.yandex.direct.ess.config.bsexport.BsExportStrategyConfig
import ru.yandex.direct.ess.logicobjects.bsexport.strategy.BsExportStrategyObject
import ru.yandex.direct.ess.logicobjects.bsexport.strategy.StrategyResourceType
import ru.yandex.direct.oneshot.oneshots.bsexport.BaseResyncOneshot
import ru.yandex.direct.oneshot.oneshots.bsexport.ResyncTableRow
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtField

private typealias StrategyId = Long

class StrategiesResyncTableRow : ResyncTableRow<StrategyId>(listOf(STRATEGY_ID)) {
    private val strategyId: Long?
        get() = valueOf(STRATEGY_ID)

    val id: Long
        get() = strategyId ?: error("ERROR: found null strategyId")

    override fun convert(): StrategyId = strategyId ?: error("ERROR: found null strategyId")

    companion object {
        private val STRATEGY_ID = YtField("strategyId", Long::class.java)
    }
}

@Component
@Approvers("pavryabov", "ruslansd", "kuvshinov", "ninazhevtyak")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class StrategiesResyncOneshot @Autowired constructor(
    ytProvider: YtProvider,
    essClient: EssClient,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val strategyRepository: StrategyTypedRepository
) : BaseResyncOneshot<StrategyId, StrategiesResyncTableRow>(ytProvider, essClient) {
    override val chunkSizeProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.RESYNC_STRATEGIES_CHUNK_SIZE)

    override val relaxTimeProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.RESYNC_STRATEGIES_RELAX_TIME)

    override val essLogicProcessName: String = BsExportStrategyConfig().logicProcessName

    override fun createEmptyYtRow(): StrategiesResyncTableRow =
        StrategiesResyncTableRow()

    override fun getLogicObjects(shard: Int, requests: List<StrategyId>): List<BaseLogicObject> {
        val ids = requests.distinct()
        val entities = getEntitiesFromRepository(ids, shard)
            .associateBy { it.id }

        return requests
            .filter { it in entities }
            .map { createLogicObject(entities.getValue(it)) }
    }

    private fun getEntitiesFromRepository(ids: Collection<Long>, shard: Int): List<BaseStrategy> =
        strategyRepository.getTyped(shard, ids)

    private fun createLogicObject(entity: BaseStrategy): BaseLogicObject =
        BsExportStrategyObject(
            entity.id, StrategyResourceType.ALL, Trace.current().spanId,
            Trace.current().service, Trace.current().method
        )

}
