package ru.yandex.direct.oneshot.oneshots.bsresync;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.oneshot.base.YtState;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.NumberConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.exceptions.ReadException;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

/**
 * Ваншот для переотправки объектов в БК по таблице из YTя.
 * Берет cid, bid, pid и приоритет из таблицы в YT и добавляет в очередь ленивой переотправки.
 * Приоритет из входных данных будет использован в первую очередь. Если во входных данных приоритет не указан, то
 * будет взят приоритет из таблицы в YT, а если и его нет, то приоретет по умолчанию.
 * https://st.yandex-team.ru/DIRECT-120888
 * <p>
 * Ваншот ожидает два обязательных входных параметра - строку с кластером YT, на котором лежит таблица
 * и строку с путем к таблице с данными, и один опциональный - приоретет (-128..127)
 */
@Component
@Approvers({"ppalex", "hmepas", "lena-san", "gerdler", "maxlog", "zhur", "kuhtich", "hrustyashko", "pavryabov"})
@Multilaunch
public class ImportToBsResyncQueueFromYtOneshot implements SimpleOneshot<BsResyncInputData, YtState> {
    private static final Logger logger = LoggerFactory.getLogger(ImportToBsResyncQueueFromYtOneshot.class);
    private static final int CHUNK_SIZE = 100_000;

    private final BsResyncService bsResyncService;
    private final YtProvider ytProvider;

    @Autowired
    public ImportToBsResyncQueueFromYtOneshot(BsResyncService bsResyncService, YtProvider ytProvider) {
        this.bsResyncService = bsResyncService;
        this.ytProvider = ytProvider;
    }

    @Override
    public YtState execute(BsResyncInputData bsResyncInputData, YtState prevState) {
        YtCluster ytCluster = YtCluster.parse(bsResyncInputData.getYtCluster());
        YtTable ytTable = new YtTable(bsResyncInputData.getTablePath());
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (prevState == null) {
            logger.info("First iteration!");
            prevState = new YtState()
                    .withNextRowFromYtTable(0L)
                    .withTotalRowCount(ytOperator.readTableRowCount(ytTable));
        }

        long rowCount = prevState.getTotalRowCount();
        long startRow = prevState.getNextRow();
        long endRow = Math.min(prevState.getNextRow() + CHUNK_SIZE, rowCount);
        if (startRow >= rowCount) {
            logger.info("Last iteration, last processed row: {}, total rows: {}", startRow, rowCount);
            return null;
        }
        try {
            BsResyncTableRow bsResyncTableRow = new BsResyncTableRow();
            List<BsResyncItem> items = new ArrayList<>();

            ytOperator.readTableByRowRange(ytTable, row -> items.add(convertRow(row, bsResyncInputData)),
                    bsResyncTableRow,
                    startRow,
                    endRow);
            processRows(items);
            logger.info("Processed {} items, from row {} to {}", items.size(), startRow, endRow);
        } catch (ReadException e) {
            logger.error("Caught an read exception: " + e.getMessage());
            throw e;
        }
        return new YtState().withNextRowFromYtTable(endRow).withTotalRowCount(rowCount);
    }

    @Override
    public ValidationResult<BsResyncInputData, Defect> validate(BsResyncInputData bsResyncInputData) {
        ItemValidationBuilder<BsResyncInputData, Defect> vb = ItemValidationBuilder.of(bsResyncInputData);

        vb.item(bsResyncInputData.getPriority(), "priority")
                .check(NumberConstraints.inRange(BsResyncItem.MIN_PRIORITY, BsResyncItem.MAX_PRIORITY));

        Set<String> setOfYtClusters = new HashSet<>();
        for (YtCluster c : YtCluster.values()) {
            setOfYtClusters.add(c.getName());
        }
        vb.item(bsResyncInputData.getYtCluster(), "ytCluster")
                .check(CommonConstraints.notNull())
                .check(CommonConstraints.inSet(setOfYtClusters));

        // если ошиблись с кластером, то проверить существование таблицы не сможем - выходим
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        YtCluster ytCluster = YtCluster.parse(bsResyncInputData.getYtCluster());
        vb.item(bsResyncInputData.getTablePath(), "tablePath")
                .check(CommonConstraints.notNull())
                .check(fromPredicate(tablePath -> ytProvider.getOperator(ytCluster).exists(new YtTable(tablePath)),
                        CommonDefects.objectNotFound()));
        return vb.getResult();
    }

    private BsResyncItem convertRow(BsResyncTableRow bsResyncTableRow, BsResyncInputData bsResyncInputData) {
        Long priority = bsResyncInputData.getPriority();
        if (priority == null) {
            priority = bsResyncTableRow.getPriority() != null ? bsResyncTableRow.getPriority() :
                    BsResyncPriority.DEFAULT.value();
        }

        return new BsResyncItem(
                priority,
                bsResyncTableRow.getCid(),
                bsResyncTableRow.getBid(),
                bsResyncTableRow.getPid()
        );
    }

    private void processRows(List<BsResyncItem> rows) {
        bsResyncService.addObjectsToResync(rows);
    }
}
