package ru.yandex.direct.oneshot.oneshots.calltracking

import org.jooq.DSLContext
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.util.RepositoryUtils.booleanFromLong
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings
import ru.yandex.direct.core.entity.calltrackingsettings.repository.mapper.CalltrackingSettingsMapper.phonesToTrackFromJson
import ru.yandex.direct.core.entity.domain.service.DomainService
import ru.yandex.direct.dbschema.ppc.Tables.CALLTRACKING_SETTINGS
import ru.yandex.direct.dbschema.ppc.Tables.CAMP_CALLTRACKING_SETTINGS
import ru.yandex.direct.dbschema.ppc.Tables.DOMAINS
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.base.ShardedOneshotWithoutInput
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch

private const val WWW_PREFIX_DOMAIN = "www."

/**
 * Вырезать в настройках коллтрекинга [WWW_PREFIX_DOMAIN] у доменов
 */
@Component
@Multilaunch
@Approvers("maxlog")
class CalltrackingOnSiteWwwDomainOneshot(
    private val dslContextProvider: DslContextProvider,
    private val domainService: DomainService
) : ShardedOneshotWithoutInput() {

    companion object {
        private val logger = LoggerFactory.getLogger(CalltrackingOnSiteWwwDomainOneshot::class.java)
    }

    override fun execute(shard: Int) {
        val dslContext = dslContextProvider.ppc(shard)

        // Достаем все настройки вместе с доменом
        val settingsToDomain = getSettingsToDomain(dslContext)

        // Достаем настройки только с www доменами
        val settingsToWwwDomain = settingsToDomain.filterValues { d -> d.startsWith(WWW_PREFIX_DOMAIN) }
        if (settingsToWwwDomain.isEmpty()) {
            logger.info("No www domains found")
            return
        }
        // Отображение: идентификатор домена -> список настроек с этим доменом
        // Пригодится, когда будем искать -- есть ли настройки с точно таким же доменом, но только без www
        val domainIdToSettings = settingsToDomain.keys.groupBy { it.domainId }

        // Отображение: домен с [WWW_PREFIX_DOMAIN] -> идентификатор этого же домена без [WWW_PREFIX_DOMAIN]
        val wwwDomainToCutWwwDomainId = getWwwDomainToCutWwwDomainId(dslContext, settingsToWwwDomain.values)

        // Собираем два отображения для обновления в БД
        // Отображение: идентификатор настроек c www -> идентификатор домена без www
        val settingIdToCutWwwDomainId = mutableMapOf<Long, Long>()
        // Отображение: идентификатор настроек с www ->  идентификатор точно таких же настроек только без www
        // Для случая, когда у одного клиента есть две абсолютно одинаковые настройки, но одна с www, другая без
        val wwwSettingIdToCutWwwSettingId = mutableMapOf<Long, Long>()

        settingsToWwwDomain.forEach { (settings, wwwDomain) ->
            // Домен без www, на который хотим поменять настройки
            val cutWwwDomainId = wwwDomainToCutWwwDomainId.getValue(wwwDomain)
            // Проверим есть ли уже настройки с таким доменом
            val cutWwwSettings = domainIdToSettings[cutWwwDomainId]
            if (cutWwwSettings == null) {
                // Таких настроек еще нет -- можно безопасно подменять домен у этих настроек на домен без www
                settingIdToCutWwwDomainId[settings.id] = cutWwwDomainId
            } else {
                // Настройки с доменом без www уже есть
                // Проверяем, принадлежат ли они тому же клиенту
                val clientId = settings.clientId
                val clientSettings = cutWwwSettings.find { it.clientId == clientId && it.domainId == cutWwwDomainId }
                if (clientSettings == null) {
                    // Именно у этого клиента нет настроек с таким же доменом без www --  безопасно подменять домен
                    settingIdToCutWwwDomainId[settings.id] = cutWwwDomainId
                } else {
                    // У этого клиента уже есть настройки с таким же доменом без www
                    if (equalsWithoutDomainId(settings, clientSettings)) {
                        // Все, кроме доменов, совпадает -- подменяем сами настройки, а не домены
                        wwwSettingIdToCutWwwSettingId[settings.id] = clientSettings.id
                    } else {
                        // Что-то не совпадает. Решения нет, нужно разбираться вручную
                        logger.warn("Found settings with cut www domain, but some fields don't match: " +
                            "${settings.id} and ${clientSettings.id}")
                    }
                }
            }
        }
        updateCalltrackingSettingsDomainIds(dslContext, settingIdToCutWwwDomainId)
        updateCalltrackingSettingIds(dslContext, wwwSettingIdToCutWwwSettingId)
        deleteUnusedCalltrackingSettings(dslContext, wwwSettingIdToCutWwwSettingId.keys)
    }

    /**
     * Получить настройки с доменами
     */
    private fun getSettingsToDomain(dslContext: DSLContext): Map<CalltrackingSettings, String> {
        return dslContext
            .select(
                CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID,
                CALLTRACKING_SETTINGS.CLIENT_ID,
                CALLTRACKING_SETTINGS.DOMAIN_ID,
                CALLTRACKING_SETTINGS.COUNTER_ID,
                CALLTRACKING_SETTINGS.PHONES_TO_TRACK,
                CALLTRACKING_SETTINGS.IS_AVAILABLE_COUNTER,
                DOMAINS.DOMAIN
            )
            .from(CALLTRACKING_SETTINGS)
            .join(DOMAINS).using(CALLTRACKING_SETTINGS.DOMAIN_ID)
            .fetchMap(
                {
                    CalltrackingSettings().apply {
                        id = it.getValue(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID)
                        clientId = ClientId.fromLong(it.getValue(CALLTRACKING_SETTINGS.CLIENT_ID))
                        domainId = it.getValue(CALLTRACKING_SETTINGS.DOMAIN_ID)
                        counterId = it.getValue(CALLTRACKING_SETTINGS.COUNTER_ID)
                        phonesToTrack = phonesToTrackFromJson(it.getValue(CALLTRACKING_SETTINGS.PHONES_TO_TRACK))
                        isAvailableCounter = booleanFromLong(it.getValue(CALLTRACKING_SETTINGS.IS_AVAILABLE_COUNTER))
                    }
                },
                {
                    it.getValue(DOMAINS.DOMAIN)
                }
            )
    }

    private fun getWwwDomainToCutWwwDomainId(dslContext: DSLContext, domains: Collection<String>): Map<String, Long> {
        val cutWwwDomains = domains.map { it.substring(WWW_PREFIX_DOMAIN.length) }
        val cutWwwDomainIds = domainService.getOrCreate(dslContext, cutWwwDomains)
        return domains
            .mapIndexed { i, d -> d to cutWwwDomainIds[i] }
            .toMap()
    }

    private fun equalsWithoutDomainId(first: CalltrackingSettings, second: CalltrackingSettings): Boolean {
        if (first.counterId != second.counterId) {
            return false
        }
        if (first.phonesToTrack.size != second.phonesToTrack.size) {
            return false
        }
        if (first.isAvailableCounter != second.isAvailableCounter) {
            return false
        }
        val firstPhones = first.phonesToTrack.mapTo(HashSet()) { it.phone }
        val secondPhones = second.phonesToTrack.mapTo(HashSet()) { it.phone }
        return firstPhones == secondPhones
    }

    private fun updateCalltrackingSettingsDomainIds(dslContext: DSLContext, settingIdToDomainId: Map<Long, Long>) {
        if (settingIdToDomainId.isEmpty()) {
            return
        }
        logger.info("Update settingId -> domainId: $settingIdToDomainId")
        val values = DSL.choose(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID).mapValues(settingIdToDomainId)
        dslContext
            .update(CALLTRACKING_SETTINGS)
            .set(CALLTRACKING_SETTINGS.DOMAIN_ID, values)
            .where(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.`in`(settingIdToDomainId.keys))
            .execute()
    }

    private fun updateCalltrackingSettingIds(dslContext: DSLContext, oldSettingIdToNew: Map<Long, Long>) {
        if (oldSettingIdToNew.isEmpty()) {
            return
        }
        logger.info("Update settingId -> settingId: $oldSettingIdToNew")
        val values = DSL.choose(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID).mapValues(oldSettingIdToNew)
        dslContext
            .update(CAMP_CALLTRACKING_SETTINGS)
            .set(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID, values)
            .where(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.`in`(oldSettingIdToNew.keys))
            .execute()
    }

    private fun deleteUnusedCalltrackingSettings(dslContext: DSLContext, settingIds: Collection<Long>) {
        if (settingIds.isEmpty()) {
            return
        }
        logger.info("Delete settingIds: $settingIds")
        dslContext
            .delete(CALLTRACKING_SETTINGS)
            .where(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.`in`(settingIds))
            .execute()
    }
}
