package ru.yandex.direct.oneshot.oneshots.calltracking

import org.jooq.DSLContext
import org.jooq.TransactionalRunnable
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect

data class InputSettings(
    val sourceId: Long,
    val targetId: Long?
)

@Component
@Multilaunch
@Approvers("maxlog")
class CalltrackingSettingsReplacerOneshot(
    private val dslContextProvider: DslContextProvider
) : ShardedOneshot<InputSettings, Void> {

    companion object {
        private val logger = LoggerFactory.getLogger(CalltrackingOnSiteWwwDomainOneshot::class.java)
    }

    override fun validate(inputData: InputSettings) =
        ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.sourceId, "sourceId")
                .check(CommonConstraints.validId())
            item(inputData.targetId, "targetId")
                .check(CommonConstraints.validId())
        }.result

    override fun execute(inputData: InputSettings, prevState: Void?, shard: Int): Void? {
        dslContextProvider.ppc(shard).transaction { configuration ->
            val dsl = configuration.dsl()
            TransactionalRunnable {
                if (inputData.targetId != null) {
                    val campaignIds = getCampaignIds(dsl, inputData.sourceId)
                    if (campaignIds.isNotEmpty()) {
                        replaceSettings(dsl, campaignIds, inputData.targetId)
                    }
                }
                deleteSettings(dsl, inputData.sourceId)
            }.run(configuration)
        }
        return null
    }

    private fun getCampaignIds(dslContext: DSLContext, settingsId: Long): List<Long> {
        val campaignIds = dslContext
            .select(Tables.CAMP_CALLTRACKING_SETTINGS.CID)
            .from(Tables.CAMP_CALLTRACKING_SETTINGS)
            .where(Tables.CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.eq(settingsId))
            .fetch(Tables.CAMP_CALLTRACKING_SETTINGS.CID)
        logger.info("Campaigns with settingsId $settingsId: $campaignIds")
        return campaignIds
    }

    private fun replaceSettings(dslContext: DSLContext, campaignIds: Collection<Long>, targetId: Long) {
        dslContext
            .update(Tables.CAMP_CALLTRACKING_SETTINGS)
            .set(Tables.CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID, targetId)
            .where(Tables.CAMP_CALLTRACKING_SETTINGS.CID.`in`(campaignIds))
            .execute()
        logger.info("Update settings, set $targetId for $campaignIds")
    }

    private fun deleteSettings(dslContext: DSLContext, sourceId: Long) {
        dslContext
            .delete(Tables.CALLTRACKING_SETTINGS)
            .where(Tables.CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.eq(sourceId))
            .execute()
        logger.info("Delete $sourceId settings")
    }
}
