package ru.yandex.direct.oneshot.oneshots.campaign

import org.jooq.DSLContext
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.CampaignsMetatype
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow
import java.util.EnumSet

data class CampaignData(
    val metatype: CampaignsMetatype,
    val opts: EnumSet<CampaignOpts>,
)

data class RemoveCampaignRecommendationsManagementOneshotInputData(
    val ytCluster: YtCluster,
    val tablePath: String,
    val sleepCoefficient: Double,
)

private class InputYtTableRow : YtTableRow(listOf(CID)) {
    companion object {
        private val CID = YtField("cid", Long::class.java)
    }

    val cid: Long
        get() = valueOf(CID)
}

@Component
@Multilaunch
@Approvers("khuzinazat", "pavelkataykin", "bratgrim")
@Retries(5)
class RemoveCampaignRecommendationsManagementOneshot constructor(
    private val dslContextProvider: DslContextProvider,
    private val shardHelper: ShardHelper,
    private val ytProvider: YtProvider,
) : SimpleOneshot<RemoveCampaignRecommendationsManagementOneshotInputData, Void> {

    companion object {
        private val logger = LoggerFactory.getLogger(this::class.java)
    }

    override fun validate(inputData: RemoveCampaignRecommendationsManagementOneshotInputData)
        = validateObject(inputData) {
        ValidateUtil.validateTableExistsInYt(
            ytProvider,
            ItemValidationBuilder.of(inputData, Defect::class.java),
            inputData.ytCluster,
            inputData.tablePath
        )
    }

    override fun execute(inputData: RemoveCampaignRecommendationsManagementOneshotInputData, prevState: Void?): Void? {
        val campaignIds = mutableListOf<Long>()
        ytProvider.getOperator(inputData.ytCluster).readTable(
            YtTable(inputData.tablePath),
            InputYtTableRow()
        ) {
            campaignIds.add(it.cid)
        }

        if (campaignIds.isEmpty()) {
            logger.error("Empty input table!")
            return null
        }

        shardHelper.groupByShard(campaignIds, ShardKey.CID).forEach { shard, shardedCampaignIds ->
            shardedCampaignIds.forEach {
                RelaxedWorker(inputData.sleepCoefficient).callAndRelax<Unit, RuntimeException> {
                    processCampaign(shard, it)
                }
            }
        }

        return null
    }

    fun processCampaign(shard: Int, campaignId: Long) {
        logger.info("Working on campaign {}", campaignId)
        dslContextProvider.ppcTransaction(shard) { conf ->
            val context = conf.dsl()
            val campaignData = getCampaignData(context, campaignId)
            if (campaignData == null) {
                logger.error("Campaign $campaignId not found")
                return@ppcTransaction
            }
            if (campaignData.metatype != CampaignsMetatype.ecom) {
                logger.info("Wrong metatype for campaign {}", campaignId)
                return@ppcTransaction
            }
            if (!campaignData.opts.contains(CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED)) {
                logger.info("No recommendations management option found for campaign {}", campaignId)
                return@ppcTransaction
            }
            logger.info("Campaign $campaignId had opts: ${campaignData.opts}")
            campaignData.opts.removeAll(setOf(CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED,
                CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED))
            setOpts(context, campaignId, campaignData.opts)
        }
    }

    private fun getCampaignData(dslContext: DSLContext, campaignId: Long): CampaignData? {
        val record = dslContext
                .select(Tables.CAMPAIGNS.OPTS, Tables.CAMPAIGNS.METATYPE)
                .from(Tables.CAMPAIGNS)
                .where(Tables.CAMPAIGNS.CID.eq(campaignId))
                .fetchOne() ?: return null
        return CampaignData(
            metatype = record.component2(),
            opts = CampaignMappings.optsFromDb(record.component1())
        )
    }

    fun setOpts(dslContext: DSLContext, campaignId: Long, optsSet: EnumSet<CampaignOpts>) {
        dslContext
            .update(Tables.CAMPAIGNS)
            .set(Tables.CAMPAIGNS.OPTS, CampaignMappings.optsToDb(optsSet))
            .set(Tables.CAMPAIGNS.LAST_CHANGE, Tables.CAMPAIGNS.LAST_CHANGE)
            .where(Tables.CAMPAIGNS.CID.eq(campaignId))
            .execute()
    }
}
