package ru.yandex.direct.oneshot.oneshots.campaign

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.container.CampaignNewMinusKeywords
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.keyword.model.Keyword
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory
import ru.yandex.direct.core.entity.minuskeywordspack.MinusKeywordsPackUtils.minusKeywordsFromJson
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.model.YtTableRow

data class InputData(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operator: Long?,
    val dryRun: Boolean?,
    val deleteKeywords: Boolean?, // производить ли удаление ключевых фраз, совпадающих с переданными минус-словами
)

data class State(
    var lastRow: Long,
)

class InputTableRow : YtTableRow(listOf(CID, MINUS_KEYWORDS)) {
    companion object {
        private val CID = YtField("cid", Long::class.java)
        private val MINUS_KEYWORDS = YtField("minus_keywords", String::class.java)
    }

    val cid: Long
        get() = valueOf(CID)

    val minusKeywords: String
        get() = valueOf(MINUS_KEYWORDS)
}

/**
 * Ваншот для восстановления минус фраз на текстовых кампаниях из Мастера кампаний.
 * Принимает на вход пары <id кампании, минус-фразы> (в виде yt-таблицы) и делает две вещи:
 * 1) удаляет из кампании и ее заявки ключевые фразы, которые пересекаются с минус-фразами на входе,
 *    если передан ключ deleteKeywords
 * 2) добавляет входные минус-фразы в минус-фразы на кампании.
 *
 * По-умолчанию ваншот ничего не меняет в базе, для настоящего прогона надо передать dryRun=true.
 * Также по-умолчанию с операциях используется uid оператора клиента кампании,
 * но есть возможность передать uid оператора во входных данных (например супера).
 *
 * В yt таблице должны быть следующие колонки: cid, minus_keywords
 */
@Component
@Multilaunch
@Approvers("khuzinazat", "mspirit", "dimitrovsd", "bratgrim")
@Retries(5)
@PausedStatusOnFail
class RestoreCampaignMinusKeywordsOneshot constructor(
    private val ytProvider: YtProvider,
    private val shardHelper: ShardHelper,
    private val campaignService: CampaignService,
    private val keywordRepository: KeywordRepository,
    private val keywordOperationFactory: KeywordOperationFactory,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val grutTransactionProvider: GrutTransactionProvider,
    @Value("\${object_api.retries}") private val grutRetries: Int,
) : SimpleOneshot<InputData, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(RestoreCampaignMinusKeywordsOneshot::class.java)
    }

    override fun validate(inputData: InputData) = validateObject(inputData) {
        property(inputData::ytCluster) {
            check(CommonConstraints.notNull())
        }
        property(inputData::tablePath) {
            check(CommonConstraints.notNull())
            check(
                Constraint.fromPredicate(
                    { ytProvider.getOperator(inputData.ytCluster).exists(YtTable(it)) },
                    CommonDefects.objectNotFound()
                ), When.isValid()
            )
        }
    }

    override fun execute(inputData: InputData, prevState: State?): State? {
        val state = prevState ?: State(0)

        val startRow = state.lastRow
        val lastRow = startRow + 1
        logger.info("Iteration starts with row $startRow")

        val cids = mutableListOf<Long>()
        val minusKeywordsList = mutableListOf<String>()
        ytProvider.getOperator(inputData.ytCluster).readTableByRowRange(
            YtTable(inputData.tablePath), {
                cids.add(it.cid)
                minusKeywordsList.add(it.minusKeywords)
            },
            InputTableRow(), startRow, lastRow
        )

        if (cids.isEmpty()) {
            logger.info("Work completed!")
            return null
        }
        val cid = cids[0]
        val minusKeywords = minusKeywordsFromJson(minusKeywordsList[0])?.toSet()
        if (minusKeywords.isNullOrEmpty()) {
            logger.warn("No minus keywords for cid=$cid")
            return State(lastRow)
        }
        val shard = shardHelper.getShardByCampaignId(cid)
        if (shard == NO_SHARD) {
            logger.warn("No shard found for campaign $cid")
            return State(lastRow)
        }
        val campaigns = campaignTypedRepository.getSafely(shard, listOf(cid), CommonCampaign::class.java)
        if (campaigns.isEmpty()) {
            logger.error("Campaign $cid not found")
            return State(lastRow)
        }
        val campaign = campaigns[0]
        if (campaign.type != CampaignType.TEXT) {
            logger.info("Skip campaign $cid with type ${campaign.type}")
            return State(lastRow)
        }

        logger.info("Processing campaign: $cid")
        val clientId = ClientId.fromLong(campaign.clientId)
        val operatorUid = inputData.operator ?: campaign.uid
        val dryRun = inputData.dryRun ?: true
        val deleteKeywords = inputData.deleteKeywords ?: false

        val keywords = keywordRepository.getKeywordsByCampaignId(shard, cid)
        val keywordsToDelete = keywords.filter { minusKeywords.contains(it.phrase) }.associate { it.id to it.phrase }
        if (keywordsToDelete.isNotEmpty()) {
            logger.info("Deleting ${keywordsToDelete.size} keywords from campaign $cid: $keywordsToDelete")
            if (!dryRun && deleteKeywords) {
                deleteKeywords(operatorUid, clientId, keywordsToDelete.keys, keywords)
            } else {
                logger.info("Keywords deletion is not requested or the run is dry, no deleting keywords")
            }
        }

        logger.info("Appending ${minusKeywords.size} minus keywords")
        if (!dryRun) {
            appendMinusKeywords(cid, operatorUid, clientId, minusKeywords, shard)
        } else {
            logger.info("Run is dry, no appending minus keywords")
        }
        logger.info("Iteration finished")
        return State(lastRow)
    }

    private fun deleteKeywords(
        operatorUid: Long,
        clientId: ClientId,
        keywordIdsToDelete: Set<Long>,
        allKeywords: List<Keyword>
    ) {
        val keywordsDeleteOperation = keywordOperationFactory.createKeywordsDeleteOperation(
            Applicability.PARTIAL, keywordIdsToDelete.toList(), operatorUid, clientId)
        val deleteKeywordsResult = keywordsDeleteOperation.prepareAndApply()
        deleteKeywordsResult.logErrors()
        val successKeywordIds = getSuccessResults(deleteKeywordsResult)
        val failKeywordIds = keywordIdsToDelete - successKeywordIds
        if (failKeywordIds.isNotEmpty()) {
            logger.error("Failed to delete keywords: $failKeywordIds")
        }
        if (successKeywordIds.isEmpty()) {
            logger.info("No keywords to delete from grut")
        } else {
            logger.info("Successfully deleted ${successKeywordIds.size} keywords: $successKeywordIds")
            val phrasesByCampaignsId = allKeywords
                .filter { successKeywordIds.contains(it.id) }
                .groupBy({ it.campaignId }, { it.phrase })
            logger.info("Deleting keywords from grut")
            grutTransactionProvider.runInRetryableTransaction(grutRetries) {
                grutUacCampaignService.deleteCampaignKeywordsAndAppendMinusKeywords(phrasesByCampaignsId, phrasesByCampaignsId)
            }
        }
    }

    private fun appendMinusKeywords(
        cid: Long,
        operatorUid: Long,
        clientId: ClientId,
        minusKeywords: Set<String>,
        shard: Int,
    ) {
        val minusKeywordsList = filterMinusKeywords(cid, minusKeywords, shard)
        if (minusKeywordsList.isEmpty()) {
            return
        }
        val appendMinusKeywordsOperation = campaignService.createAppendMinusKeywordsOperation(
            listOf(CampaignNewMinusKeywords(cid, minusKeywordsList)),
            operatorUid,
            clientId
        )
        val appendMinusKeywordsResult = appendMinusKeywordsOperation.prepareAndApply()
        appendMinusKeywordsResult.logErrors()
    }

    fun filterMinusKeywords(
        cid: Long,
        minusKeywords: Set<String>,
        shard: Int,
    ) : List<String> {
        // перечитываем ключевики после возможного удаления
        val keywords = keywordRepository.getKeywordsByCampaignId(shard, cid)
        val phrases = mutableSetOf<String>()
        keywords.forEach {
            phrases.add(it.phrase)
            phrases.add(it.normPhrase)
        }
        return minusKeywords
            .filter { !phrases.contains(it) }
    }

    private fun getSuccessResults(massResult: MassResult<Long>?): Set<Long> {
        return if (massResult?.result == null) {
            emptySet()
        } else massResult.result
            .filterNotNull()
            .filter { it.isSuccessful }
            .mapNotNull { it.result }
            .toSet()
    }

    private fun MassResult<Long>.logErrors(action: String = "action") {
        if (!this.isSuccessful) {
            logger.error("Result of $action was unsuccessful: {0}", this.validationResult.flattenErrors())
        }
    }
}
