package ru.yandex.direct.oneshot.oneshots.campaign

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.validation.StrategyDefects.strategyNotFound
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyModifyRepository
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

private const val CAMPAIGNS_NUMBER_LIMIT = 100

@Component
@Multilaunch
@Approvers("buhter", "ali-al", "munira")
class SetPackageStrategyCampaignsOneshot(
    private val shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
    private val campaignModifyRepository: CampaignModifyRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val strategyModifyRepository: StrategyModifyRepository,
    private val strategyTypedRepository: StrategyTypedRepository,
) : SimpleOneshot<CreateCampaignsWithPackageStrategyInputData, Void> {
    override fun validate(inputData: CreateCampaignsWithPackageStrategyInputData) = validateObject(inputData) {
        val clientId = ClientId.fromLong(inputData.clientId)
        val shard = shardHelper.getShardByClientId(clientId)

        property(inputData::clientId)
            .check(CommonConstraints.validId(), When.isValid())
            .check(Constraint.fromPredicate({ shardHelper.isExistentClientId(it) }, CommonDefects.objectNotFound()), When.isValid())

        property(inputData::strategyId)
            .check(CommonConstraints.validId(), When.isValid())
            .check(
                Constraint.fromPredicate(
                    { strategyId: Long -> strategyTypedRepository.getTyped(shard, listOf(strategyId)).isNotEmpty() },
                    strategyNotFound()
                ), When.isValidAnd(When.isTrue(shard != -1))
            )

        listProperty(inputData::campaignIds)
            .check(CommonConstraints.notNull())
            .check(CollectionConstraints.notEmptyCollection())
            .check(CollectionConstraints.maxListSize(CAMPAIGNS_NUMBER_LIMIT))
            .checkEach(CommonConstraints.validId(), When.isValid())

    }

    override fun execute(inputData: CreateCampaignsWithPackageStrategyInputData, prevState: Void?): Void? {
        val shard = shardHelper.getShardByClientId(ClientId.fromLong(inputData.clientId))
        updatePackageStrategy(shard, inputData.strategyId, inputData.campaignIds)
        return null
    }

    private fun updatePackageStrategy(shard: Int, strategyId: Long, campaignIds: List<Long>) {
        dslContextProvider.ppcTransaction(shard) { conf ->
            val dslContext = conf.dsl()

            val strategy = strategyTypedRepository.getTyped(shard, listOf(strategyId))[0] as CommonStrategy
            val campaigns = campaignTypedRepository.getTypedCampaigns(dslContext, campaignIds).map { it as CampaignWithPackageStrategy }

            val strategyChanges = ModelChanges(strategy.id, CommonStrategy::class.java)
                .process(campaignIds, CommonStrategy.CIDS)
                .applyTo(strategy)
            strategyModifyRepository.updateStrategiesTable(shard, listOf(strategyChanges))

            val campaignChanges = campaigns.map { campaign ->
                ModelChanges(campaign.id, CampaignWithPackageStrategy::class.java)
                    .process(strategyId, CampaignWithPackageStrategy.STRATEGY_ID)
                    .applyTo(campaign)
            }
            campaignModifyRepository.updateCampaignsTable(
                dslContext,
                campaignChanges
            )
        }
    }
}

data class CreateCampaignsWithPackageStrategyInputData(
    val clientId: Long,
    val strategyId: Long,
    val campaignIds: List<Long>,
)
