package ru.yandex.direct.oneshot.oneshots.campaign

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect

data class Input(
    val shard: Int,
    val walletIds: List<Long>
)

/**
 * Oneshot предназначен для точечного отключения кошельков дубликатов
 */

@Component
@Multilaunch
@Approvers("a-dubov", "bakin", "buhter")
class SetWalletsStatusEmptyToTrue @Autowired constructor(
    private val campaignRepository: CampaignRepository
) : SimpleOneshot<Input, Void> {

    companion object {
        private val logger = LoggerFactory.getLogger(SetWalletsStatusEmptyToTrue::class.java)
    }

    override fun validate(inputData: Input) =
        ItemValidationBuilder.of(inputData, Defect::class.java).apply {
            item(inputData.walletIds, "walletIds")
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.minListSize(1))
                .check(CollectionConstraints.maxListSize(10))
        }.result

    override fun execute(inputData: Input, prevState: Void?): Void? {
        val shard = inputData.shard
        val walletIds = inputData.walletIds

        logger.info(String.format("Setting statusEmpty = Yes for wallets: %s", walletIds.toString()))

        campaignRepository.setStatusEmptyToYes(shard, walletIds)

        logger.info(String.format("Shard: %d: SetWalletsStatusEmptyToTrue FINISH", shard))

        return null
    }

}
