package ru.yandex.direct.oneshot.oneshots.campaignsarchive;

import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.product.service.ProductService.ZEN_PRODUCT_ID;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Component
@Multilaunch
@Approvers({"a-dubov", "ppalex", "khuzinazat", "maxlog"})
public class ZenCampaignsArchiveOneshot implements SimpleOneshot<ZenCampaignsArchiveInputData, Void> {
    private final CampaignRepository campaignRepository;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public ZenCampaignsArchiveOneshot(CampaignRepository campaignRepository,
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ValidationResult<ZenCampaignsArchiveInputData, Defect> validate(
            ZenCampaignsArchiveInputData data
    ) {
        ItemValidationBuilder<ZenCampaignsArchiveInputData, Defect> ivb = ItemValidationBuilder.of(data);
        ivb.item(data.getCampaignIds(), "номера кампаний")
                .check(notNull())
                .check(eachNotNull(), When.isValid())
                .checkBy(this::validateCampaignIds, When.isValid());
        return ivb.getResult();
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<List<Long>, Defect> validateCampaignIds(List<Long> campaignIds) {
        var campaignsByIds = shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedByDefault()
                .stream()
                .mapKeyValue(campaignRepository::getCampaigns)
                .flatMap(StreamEx::of)
                .toMap(Campaign::getId, c -> c);
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(campaignIds, Defect.class)
                .checkEach(validId())
                .checkEachBy(cid -> validateCampaignId(cid, campaignsByIds), When.isValid());
        return lvb.getResult();
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<Long, Defect> validateCampaignId(Long campaignId, Map<Long, Campaign> campaignsByIds) {
        ItemValidationBuilder<Long, Defect> ivb = ItemValidationBuilder.of(campaignId);
        ivb.checkByFunction(cid -> {
            if (!campaignsByIds.containsKey(cid)) {
                return objectNotFound();
            }
            var camp = campaignsByIds.get(cid);
            if (camp.getSource() != CampaignSource.ZEN) { // а продукт не проверить, т. к. его нет в модели
                return inconsistentState();
            }
            return null;
        });
        return ivb.getResult();
    }

    @Nullable
    @Override
    public Void execute(ZenCampaignsArchiveInputData inputData, @Nullable Void state) {
        var campaignIds = inputData.getCampaignIds();
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedByDefault()
                .forEach(this::updateInShard);
        return null;
    }

    private void updateInShard(int shard, List<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.ARCHIVED, CampaignsArchived.Yes)
                .set(CAMPAIGNS.STATUS_SHOW, CampaignsStatusshow.No)
                .where(CAMPAIGNS.CID.in(campaignIds)
                        .and(CAMPAIGNS.PRODUCT_ID.notEqual(ZEN_PRODUCT_ID)))
                .execute();
    }
}
