package ru.yandex.direct.oneshot.oneshots.campaignsitemonitoring

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.oneshot.base.ShardedYtOneshot
import ru.yandex.direct.oneshot.base.YtInputData
import ru.yandex.direct.oneshot.base.YtState
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import kotlin.math.min

class InputData(
    var enable: Boolean = true
) : YtInputData()

/**
 * Ваншот для включения/выключения мониторинга сайта (поле camp_options.statusMetricaControl)
 * и переотправки в БК.
 */
@Component
@Multilaunch
@Approvers("bratgrim", "gerdler", "khuzinazat")
class SetCampaignSiteMonitoringOneshot @Autowired constructor(
    ytProvider: YtProvider,
    private val campaignRepository: CampaignRepository
) : ShardedYtOneshot<InputData, YtState>(ytProvider) {
    companion object {
        private val logger = LoggerFactory.getLogger(SetCampaignSiteMonitoringOneshot::class.java)
        private const val CHUNK_SIZE = 1000L
    }

    override fun execute(inputData: InputData, prevState: YtState?, shard: Int): YtState? {
        val ytCluster = YtCluster.parse(inputData.ytCluster)
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator = ytProvider.getOperator(ytCluster)

        if (prevState == null) {
            logger.info("First iteration, shard {}!", shard)
            return YtState()
                .withNextRowFromYtTable(0L)
                .withTotalRowCount(ytOperator.readTableRowCount(ytTable))
        }

        val rowCount = prevState.totalRowCount
        val startRow = prevState.nextRow
        val endRow = min(prevState.nextRow + CHUNK_SIZE, rowCount)
        if (startRow >= rowCount) {
            logger.info("Last iteration, shard: {}, last processed row: {}, total rows: {}", shard, startRow, rowCount)
            return null
        }
        try {
            val tableRow = CampaignWithShardTableRow()
            val campaignIds = mutableListOf<Long>()
            ytOperator.readTableByRowRange(
                ytTable, { row: CampaignWithShardTableRow ->
                if (shard == row.shard) {
                    campaignIds.add(row.campaignId)
                }
            },
                tableRow,
                startRow,
                endRow
            )
            val updatedCount = campaignRepository.updateHasSiteMonitoring(shard, campaignIds, inputData.enable)
            logger.info("Updated {} rows of {} ids from row {} to {}, shard: {}",
                updatedCount, campaignIds.size, startRow, endRow, shard)
        } catch (e: RuntimeException) {
            logger.error("Caught exception on shard {}: {}", shard, e.message)
            throw e
        }
        return YtState().apply {
            nextRow = endRow
            totalRowCount = rowCount
        }
    }
}
