package ru.yandex.direct.oneshot.oneshots.campaignsource;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource;
import ru.yandex.direct.oneshot.base.ShardedYtOneshot;
import ru.yandex.direct.oneshot.base.YtInputData;
import ru.yandex.direct.oneshot.base.YtState;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;


/**
 * Ваншот для изменения source кампаний по таблице из YTя.
 * Берет shard, cid и source из таблицы в YT, в таблице campaigns по ключу cid в шарде shard меняет source.
 * https://st.yandex-team.ru/DIRECT-140609
 * <p>
 * Ваншот ожидает два обязательных входных параметра - строку с кластером YT, на котором лежит таблица
 * и строку с путем к таблице с данными
 */
@Component
@Approvers({"a-dubov", "gerdler", "khuzinazat"})
@Multilaunch
public class UpdateCampaignSourceOneshot extends ShardedYtOneshot<YtInputData, YtState> {

    private static final Logger logger = LoggerFactory.getLogger(UpdateCampaignSourceOneshot.class);
    private static final long CHUNK_SIZE = 100;

    private final CampaignRepository campaignRepository;

    public UpdateCampaignSourceOneshot(
            YtProvider ytProvider,
            CampaignRepository campaignRepository
    ) {
        super(ytProvider);
        this.campaignRepository = campaignRepository;
    }

    @Nullable
    @Override
    public YtState execute(YtInputData inputData, YtState prevState,
                           int shard) {
        YtCluster ytCluster = YtCluster.parse(inputData.getYtCluster());
        YtTable ytTable = new YtTable(inputData.getTablePath());
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (prevState == null) {
            logger.info("First iteration, shard {}!", shard);
            prevState = new YtState()
                    .withNextRowFromYtTable(0L)
                    .withTotalRowCount(ytOperator.readTableRowCount(ytTable));
        }

        long rowCount = prevState.getTotalRowCount();
        long startRow = prevState.getNextRow();
        long endRow = Math.min(prevState.getNextRow() + CHUNK_SIZE, rowCount);
        if (startRow >= rowCount) {
            logger.info("Last iteration, shard: {}, last processed row: {}, total rows: {}", shard, startRow, rowCount);
            return null;
        }
        try {
            CampaignWithSourceTableRow tableRow = new CampaignWithSourceTableRow();
            Map<CampaignsSource, Set<Long>> campaignIdsWithSources = new HashMap<>();

            ytOperator.readTableByRowRange(ytTable,row -> {
                        if (shard != row.getShard()) {
                            return;
                        }
                        CampaignsSource source = CampaignsSource.valueOf(row.getSource());
                        Set<Long> campaignIds = campaignIdsWithSources.get(source);
                        if (campaignIds == null) {
                            campaignIds = new HashSet<>();
                            campaignIdsWithSources.put(source, campaignIds);
                        }
                        campaignIds.add(row.getCampaignId());
                    },
                    tableRow,
                    startRow,
                    endRow);

            int idCount = StreamEx.ofValues(campaignIdsWithSources)
                    .mapToInt(Set::size)
                    .sum();

            int updatedCount = EntryStream.of(campaignIdsWithSources)
                    .mapToInt(entry -> campaignRepository
                            .updateCampaignSource(shard, entry.getValue(), entry.getKey())
                    )
                    .sum();

            logger.info("Update {} campaigns of {} ids, from row {} to {}, shard: {}", updatedCount, idCount, startRow, endRow, shard);
        } catch (RuntimeException e) {
            logger.error("Caught exception on shard " + shard + ": " + e.getMessage());
            throw e;
        }
        return new YtState().withNextRowFromYtTable(endRow).withTotalRowCount(rowCount);
    }

}
