package ru.yandex.direct.oneshot.oneshots.campaignstrategy

import one.util.streamex.StreamEx
import org.jooq.DSLContext
import org.jooq.Field
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapper
import ru.yandex.direct.jooqmapper.JooqMapperBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders
import ru.yandex.direct.jooqmapper.read.FieldValues
import ru.yandex.direct.jooqmapper.read.ReaderBuilders
import ru.yandex.direct.jooqmapper.write.WriterBuilders
import ru.yandex.direct.jooqmapperhelper.UpdateHelper
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.base.ShardedOneshotWithoutInput
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.utils.FunctionalUtils
import ru.yandex.direct.utils.JsonUtils
import java.time.LocalDate
import java.util.stream.Collectors

@Component
@Multilaunch
@Approvers("andreypav")
class AddPayForConvensationFlagCpvStrategyOneShot constructor(
    private val dslContextProvider: DslContextProvider,
    private val campaignRepository: CampaignRepository)
    : ShardedOneshotWithoutInput() {

    companion object {
        private val logger = LoggerFactory.getLogger(AddPayForConvensationFlagCpvStrategyOneShot::class.java)
    }

    override fun execute(shard: Int) {
        logger.info(String.format("Shard: %d: AddPayForConvensationFlagCpvStrategyOneShot START", shard))

        val dslContext = dslContextProvider.ppc(shard)
        val localStrategyDataList = getCampaignsStrategyData(dslContext)
        // выкидываем кампании со где в стратегии уже стоит поле payForConversion = 1
        // и которые уже закончились (кроме автопродления)
        val resultStrategyDataList = localStrategyDataList.filter {
            !(it.strategyData.payForConversion ?: false) && //payForConversion выключен или не указан
            ((it.strategyData.finish?.isAfter(LocalDate.now()) ?: true) || //время окончания кампании в будущем
                it.strategyData.autoProlongation?:0 > 0 && it.statusActive)  //автопролонгация выключена или не указана
        }
        resultStrategyDataList.forEach { it.strategyData.payForConversion = true }

        saveCampaignsStrategyData(dslContext, resultStrategyDataList)
        resendToBs(dslContext, resultStrategyDataList)

        logger.info(String.format("Shard: %d: AddPayForConvensationFlagCpvStrategyOneShot FINISH", shard))
    }

    // достаем id корректировок в которых требуются изменения, а также данные, которые подскажут какие именно
    // изменения
    private fun getCampaignsStrategyData(dslContext: DSLContext): List<LocalStrategyData> {
        return dslContext
            .select(CAMPAIGNS.CID, CAMPAIGNS.STRATEGY_DATA, CAMPAIGNS.STATUS_ACTIVE)
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.STRATEGY_NAME.eq(StrategyName.toSource(StrategyName.AUTOBUDGET_AVG_CPV))
                .or(CAMPAIGNS.STRATEGY_NAME.eq(StrategyName.toSource(StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD))))
            .fetch {
                LocalStrategyData(it.getValue(CAMPAIGNS.CID),
                    it.getValue(CAMPAIGNS.STATUS_ACTIVE) == CampaignsStatusactive.Yes,
                    JsonUtils.fromJson(it.getValue(CAMPAIGNS.STRATEGY_DATA), StrategyData::class.java))
            }
    }

    private fun saveCampaignsStrategyData(dslContext: DSLContext,
                                          strategyDataList: List<LocalStrategyData>) {
        logger.info(String.format("For update: %d campaigns", strategyDataList.size))
        var counter:Int = 0;
        StreamEx.of(strategyDataList).map {
            val dbStrategy = DbStrategy()
            dbStrategy.withStrategyData(it.strategyData)
            val oldDbStrategy = DbStrategy();
            val campaign: CampaignWithStrategy = CpmBannerCampaign().withId(it.cid).withStrategy(oldDbStrategy)
            val r = ModelChanges(it.cid, CampaignWithStrategy::class.java)
                .process(dbStrategy, CampaignWithStrategy.STRATEGY)
                .applyTo(campaign)
            counter += updateCampaignsStrategyData(dslContext, r)

        }.toList()
        logger.info(String.format("Updated: %d campaigns", counter))
    }

    private fun updateCampaignsStrategyData(dslContext: DSLContext,
                                            appliedChanges: AppliedChanges<CampaignWithStrategy>): Int {
        return UpdateHelper(dslContext, CAMPAIGNS.CID)
            .processUpdate(createCampaignWithStrategyMapper(), appliedChanges)
            .execute()
    }

    private fun createCampaignWithStrategyMapper(): JooqMapper<CampaignWithStrategy>? {
        return JooqMapperBuilder.builder<CampaignWithStrategy>()
            .map(ReaderWriterBuilders.property(CampaignWithStrategy.ID, CAMPAIGNS.CID))
            .writeField(CAMPAIGNS.STRATEGY_DATA,
                WriterBuilders.fromProperty(CampaignWithStrategy.STRATEGY)
                    .by { strategy: DbStrategy -> CampaignMappings.strategyDataToDb(strategy.strategyData) }) // ppc.camp_options
            .readProperty(CampaignWithStrategy.STRATEGY,
                ReaderBuilders.fromFields(listOf<Field<*>>(CAMPAIGNS.STRATEGY_DATA))
                    .by { fields: FieldValues? -> CampaignConverter.strategyFromDb(fields) })
            .build()
    }

    private fun resendToBs(dslContext: DSLContext,
                           commonDataList: List<LocalStrategyData>) {

        val affectedCampaigns: List<Long> = FunctionalUtils.mapList(commonDataList) { it.cid }
        if (!affectedCampaigns.isEmpty()) {
            campaignRepository.updateStatusBsSynced(dslContext, affectedCampaigns, StatusBsSynced.NO)
            logger.info(String.format("Updated and sent to BS: %d campaigns: %s",affectedCampaigns.size,
                affectedCampaigns.stream().map{it.toString()}.collect(Collectors.joining(", "))))
        }
    }
}

class LocalStrategyData (
    val cid: Long,
    val statusActive: Boolean,
    val strategyData: StrategyData)
