package ru.yandex.direct.oneshot.oneshots.campaigntimetarget

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.oneshot.base.ShardedYtOneshot
import ru.yandex.direct.oneshot.base.YtInputData
import ru.yandex.direct.oneshot.base.YtState
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import kotlin.math.min

/**
 * Ваншот для установки timeTarget кампаний в NULL, у которых сейчас он ';p:'
 * https://st.yandex-team.ru/DIRECT-139137
 *
 * Ваншот ожидает два обязательных входных параметра - строку с кластером YT, на котором лежит таблица
 * и строку с путем к таблице с данными
 */
@Component
@Multilaunch
@Approvers("gerdler", "ammsaid", "khuzinazat")
class SetDefaultCampaignTimeTargetOneshot @Autowired constructor(
    ytProvider: YtProvider,
    private val campaignRepository: CampaignRepository
) : ShardedYtOneshot<YtInputData, YtState>(ytProvider) {
    companion object {
        private val logger = LoggerFactory.getLogger(SetDefaultCampaignTimeTargetOneshot::class.java)
        private const val CHUNK_SIZE = 100L
    }

    override fun execute(inputData: YtInputData, prevState: YtState?, shard: Int): YtState? {
        val ytCluster = YtCluster.parse(inputData.ytCluster)
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator = ytProvider.getOperator(ytCluster)

        if (prevState == null) {
            logger.info("First iteration, shard {}!", shard)
            return YtState()
                .withNextRowFromYtTable(0L)
                .withTotalRowCount(ytOperator.readTableRowCount(ytTable))
        }

        val rowCount = prevState.totalRowCount
        val startRow = prevState.nextRow
        val endRow = min(prevState.nextRow + CHUNK_SIZE, rowCount)
        if (startRow >= rowCount) {
            logger.info("Last iteration, shard: {}, last processed row: {}, total rows: {}", shard, startRow, rowCount)
            return null
        }
        try {
            val tableRow = CampaignTimeTargetTableRow()
            val campaignIds = mutableListOf<Long>()
            ytOperator.readTableByRowRange(
                ytTable, { row: CampaignTimeTargetTableRow ->
                    if (shard == row.shard) {
                        if (row.timeTarget != ";p:") {
                            logger.warn("Time target for cid={} is '{}' (expected ';p:')",
                                row.campaignId, row.timeTarget)
                        } else {
                            campaignIds.add(row.campaignId)
                        }
                    }
                },
                tableRow,
                startRow,
                endRow
            )
            val updatedCount = campaignRepository.setDefaultCampaignTimeTarget(shard, campaignIds)
            logger.info("Updated {} campaigns of {} ids from row {} to {}, shard: {}",
                updatedCount, campaignIds.size, startRow, endRow, shard)
        } catch (e: RuntimeException) {
            logger.error("Caught exception on shard {}: {}", shard, e.message)
            throw e
        }
        return YtState().apply {
            nextRow = endRow
            totalRowCount = rowCount
        }
    }
}
