package ru.yandex.direct.oneshot.oneshots.change_permalink_to_head

import org.jooq.DSLContext
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository
import ru.yandex.direct.core.entity.organization.model.Organization
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository
import ru.yandex.direct.core.entity.organizations.service.OrganizationService.DEFAULT_LANGUAGE
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType
import ru.yandex.direct.dbutil.exception.RollbackException
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.oneshots.change_permalink_to_head.repository.BannerPermalinksOneshotRepository
import ru.yandex.direct.oneshot.oneshots.change_permalink_to_head.repository.CampaignPermalinksOneshotRepository
import ru.yandex.direct.oneshot.oneshots.change_permalink_to_head.repository.ClientPhoneOneshotRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.organizations.swagger.OrganizationsClient
import ru.yandex.direct.organizations.swagger.model.PubApiCompanyData
import ru.yandex.direct.organizations.swagger.model.PubApiExtraCompanyField
import ru.yandex.direct.telephony.client.TelephonyClient
import ru.yandex.direct.telephony.client.TelephonyClientException
import ru.yandex.direct.telephony.client.model.TelephonyPhoneRequest
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

/**
 * Ваншот для изменения пермалинка у клиентов на головоной пермалинк
 *
 * Входные данные:
 * permalinkId - id пермалинка, который будет меняться на головной
 * onlyClientIds - меняет только для выбранных клиентов. Если не указан - меняется у всех (у кого этот пермалинк)
 */

data class InputData(
    val permalinkId: Long,
    val onlyClientIds: List<Long>? = null,
)

@Component
@Multilaunch
@Approvers("mexicano", "ppalex", "zhur", "maxlog", "kalchevskaya", "maxlog")
class ChangePermalinkToHeadOneshot : SimpleOneshot<InputData, Void> {

    private val organizationRepository: OrganizationRepository
    private val shardHelper: ShardHelper
    private val telephonyClient: TelephonyClient
    private val organizationsClient: OrganizationsClient
    private val clientPhoneOneshotRepository: ClientPhoneOneshotRepository
    private val bannerPermalinksOneshotRepository: BannerPermalinksOneshotRepository
    private val campaignPermalinksOneshotRepository: CampaignPermalinksOneshotRepository
    private val dslContextProvider: DslContextProvider
    private val clientPhoneRepository: ClientPhoneRepository
    private val limitBannerPermalink: Int

    @Autowired
    constructor(
        organizationRepository: OrganizationRepository,
        shardHelper: ShardHelper,
        telephonyClient: TelephonyClient,
        organizationsClient: OrganizationsClient,
        clientPhoneOneshotRepository: ClientPhoneOneshotRepository,
        bannerPermalinksOneshotRepository: BannerPermalinksOneshotRepository,
        campaignPermalinksOneshotRepository: CampaignPermalinksOneshotRepository,
        dslContextProvider: DslContextProvider,
        clientPhoneRepository: ClientPhoneRepository
    ) {
        this.organizationRepository = organizationRepository
        this.shardHelper = shardHelper
        this.telephonyClient = telephonyClient
        this.organizationsClient = organizationsClient
        this.clientPhoneOneshotRepository = clientPhoneOneshotRepository
        this.bannerPermalinksOneshotRepository = bannerPermalinksOneshotRepository
        this.campaignPermalinksOneshotRepository = campaignPermalinksOneshotRepository
        this.dslContextProvider = dslContextProvider
        this.clientPhoneRepository = clientPhoneRepository
        this.limitBannerPermalink = 2_000
    }

    /**
     * Используется для тестов
     */
    constructor(
        organizationRepository: OrganizationRepository,
        shardHelper: ShardHelper,
        telephonyClient: TelephonyClient,
        organizationsClient: OrganizationsClient,
        clientPhoneOneshotRepository: ClientPhoneOneshotRepository,
        bannerPermalinksOneshotRepository: BannerPermalinksOneshotRepository,
        campaignPermalinksOneshotRepository: CampaignPermalinksOneshotRepository,
        dslContextProvider: DslContextProvider,
        clientPhoneRepository: ClientPhoneRepository,
        limitBannerPermalink: Int,
    ) {
        this.organizationRepository = organizationRepository
        this.shardHelper = shardHelper
        this.telephonyClient = telephonyClient
        this.organizationsClient = organizationsClient
        this.clientPhoneOneshotRepository = clientPhoneOneshotRepository
        this.bannerPermalinksOneshotRepository = bannerPermalinksOneshotRepository
        this.campaignPermalinksOneshotRepository = campaignPermalinksOneshotRepository
        this.dslContextProvider = dslContextProvider
        this.clientPhoneRepository = clientPhoneRepository
        this.limitBannerPermalink = limitBannerPermalink
    }

    companion object {
        private val logger = LoggerFactory.getLogger(ChangePermalinkToHeadOneshot::class.java)
    }

    override fun validate(inputData: InputData) =
        validateObject(inputData) {
            property(inputData::permalinkId)
                .check(CommonConstraints.notNull())
                .check(NumberConstraints.greaterThan(0L))
                .check(
                    Constraint.fromPredicate(
                        { permalinkId -> isOrganizationHeadExist(permalinkId) }, CommonDefects.objectNotFound()
                    ), When.isValid()
                )
            if (inputData.onlyClientIds != null) {
                listProperty(inputData::onlyClientIds)
                    .check(CollectionConstraints.notEmptyCollection())
                    .checkEach(NumberConstraints.greaterThan(0L))
                    .checkEach(
                        Constraint.fromPredicate(
                            { clientId -> isClientExists(clientId!!) }, CommonDefects.objectNotFound()
                        ), When.isValid()
                    )
            }
        }

    private fun isOrganizationHeadExist(permalinkId: Long): Boolean {
        val organizationInfo = organizationsClient.getSingleOrganizationInfo(
            permalinkId,
            setOf(PubApiExtraCompanyField.HEAD_PERMALINK.value),
            DEFAULT_LANGUAGE,
            null
        )
        if (organizationInfo == null || organizationInfo.headPermalink == null) {
            logger.error("Can't find organization by permalinkId = $permalinkId")
            return false
        }

        val headOrganizationInfo = organizationsClient.getSingleOrganizationInfo(
            organizationInfo.headPermalink,
            emptySet(),
            DEFAULT_LANGUAGE,
            null
        )
        if (headOrganizationInfo == null) {
            logger.error("Can't find head organization by permalinkId = ${organizationInfo.headPermalink}")
        }
        return headOrganizationInfo != null
    }

    private fun isClientExists(clientId: Long): Boolean {
        return shardHelper.isExistentClientId(clientId)
    }

    override fun execute(inputData: InputData, prevState: Void?): Void? {
        val organizationInfo = organizationsClient.getSingleOrganizationInfo(
            inputData.permalinkId,
            setOf(PubApiExtraCompanyField.HEAD_PERMALINK.value),
            DEFAULT_LANGUAGE,
            null
        )
        checkNotNull(organizationInfo)

        val organizationFields = setOf(
            PubApiExtraCompanyField.CHAIN.value,
            PubApiExtraCompanyField.METRIKA_DATA.value,
            PubApiExtraCompanyField.PUBLISHING_STATUS.value
        )
        val headOrganizationInfo = organizationsClient.getSingleOrganizationInfo(
            organizationInfo.headPermalink,
            organizationFields,
            DEFAULT_LANGUAGE,
            null
        )
        checkNotNull(headOrganizationInfo)

        val onlyClientIdsSet = inputData.onlyClientIds?.toSet()

        for (shard in shardHelper.dbShards()) {

            logger.info("Start change permalinks in $shard shard")
            val clientIdToOrganization =
                organizationRepository
                    .getOrganizationsByPermalinkIds(shard, setOf(inputData.permalinkId))
                    .values
                    .flatten()
                    .associateBy { it.clientId }

            val clientIdToHeadOrganization =
                organizationRepository
                    .getOrganizationsByPermalinkIds(shard, setOf(headOrganizationInfo.id))
                    .values
                    .flatten()
                    .associateBy { it.clientId }

            clientIdToOrganization.keys.forEach {

                // Если меняем только у выбранного клиента
                if (!onlyClientIdsSet.isNullOrEmpty() && !onlyClientIdsSet.contains(it.asLong())) {
                    return@forEach
                }

                changeClientPermalinks(
                    shard,
                    it,
                    inputData.permalinkId,
                    clientIdToOrganization[it]!!,
                    clientIdToHeadOrganization[it],
                    headOrganizationInfo
                )
            }
        }
        return null
    }

    /**
     * Меняем пермалинки у клиента, удаляем старую организацию и добавляем (если нет) новую
     *
     * @param permalinkId - id пермалинка, который заменяем
     * @param organization - организация, которую заменяем
     * @param newOrganization - организация, которой заменяем. Если ее нет - создаем новую
     * @param newOrganizationInfo - информация о новой организации (которой заменяем)
     */
    private fun changeClientPermalinks(
        shard: Int,
        clientId: ClientId,
        permalinkId: Long,
        organization: Organization,
        newOrganization: Organization?,
        newOrganizationInfo: PubApiCompanyData,
    ) {
        val newChainId = if (newOrganizationInfo.chain != null) newOrganizationInfo.chain.id else null

        try {
            dslContextProvider.ppcTransaction(shard) {
                val dslContext: DSLContext = it.dsl()

                val clientPhonesChanged = changePermalinkForClientPhones(
                    shard,
                    dslContext,
                    clientId,
                    permalinkId,
                    newOrganizationInfo
                )

                val campaignPermalinksChanged = changePermalinkForCampaignPermalinks(
                    shard,
                    dslContext,
                    clientId,
                    permalinkId,
                    newOrganizationInfo.id,
                    newChainId
                )

                val bannerPermalinksChanged = changePermalinkForBannerPermalinks(
                    shard,
                    dslContext,
                    clientId,
                    permalinkId,
                    newOrganizationInfo.id,
                    newChainId
                )

                logger.info("$clientPhonesChanged client_phones, $campaignPermalinksChanged campaign_permalinks and" +
                    " $bannerPermalinksChanged banner_permalinks were changed in $shard shard for client = $clientId")

                // Если головной организации нет у клиента - создаем
                if (newOrganization == null) {
                    val createOrganization =
                        Organization()
                            .withChainId(newChainId)
                            .withClientId(clientId)
                            .withPermalinkId(newOrganizationInfo.id)
                            .withStatusPublish(organization.statusPublish)

                    logger.info("shard = $shard, clientId = $clientId. Create new organization = $createOrganization")
                    organizationRepository.addOrUpdateOrganizations(dslContext, setOf(createOrganization))
                }
            }
        } catch (e: RollbackException) {
            logger.error("shard = $shard, clientId = $clientId. Failed to change permalink")
        }
    }

    /**
     * Меняет пермалинки у client_phones клиента и отправляет в Телефонию новые данные
     *
     * @param permalinkId - id пермалинка, который заменяем
     * @param newOrganizationInfo - информация о новой организации (которой заменяем)
     */
    private fun changePermalinkForClientPhones(
        shard: Int,
        dslContext: DSLContext,
        clientId: ClientId,
        permalinkId: Long,
        newOrganizationInfo: PubApiCompanyData,
    ): Int {
        val clientPhones = clientPhoneRepository.getAllClientOrganizationPhones(dslContext, clientId, setOf(permalinkId))
        if (clientPhones.isEmpty()) {
            return 0
        }
        logger.info("Change ${clientPhones.size} client_phone rows in $shard shard for client = $clientId")

        val newCounterId = newOrganizationInfo.metrikaData?.counter?.toLong()

        // Проверяем счетчик. Для коллтрекинга наличие счётчика критично.
        val telephonyClientPhones = clientPhones.filter { it.phoneType == ClientPhoneType.TELEPHONY }
        if (telephonyClientPhones.isNotEmpty() && newCounterId == null) {
            logger.error("shard = $shard, clientId = $clientId, newPermalinkId = ${newOrganizationInfo.id}. " +
                "newCounterId must not be null")
            // Откатываем изменения транзакции
            throw RollbackException()
        }

        clientPhones.forEach {
            logger.info("shard = $shard, clientId = $clientId. Change client_phones " +
                "with client_phone_id = ${it.id}, " +
                "permalink_id from $permalinkId to ${newOrganizationInfo.id}, " +
                "counter_id from ${it.counterId} to $newCounterId")
        }

        val countUpdated = clientPhoneOneshotRepository.changePermalinkByClientPhonePermalinkIds(
            dslContext,
            clientId,
            clientPhones.map { it.id },
            permalinkId,
            newOrganizationInfo.id,
            newCounterId
        )

        telephonyClientPhones
            .forEach {
                sendUpdatedDataToTelephony(
                    clientId,
                    it,
                    newOrganizationInfo.id,
                    newCounterId!!
                )
            }

        return countUpdated
    }

    /**
     * Меняет пермалинки у campaign_permalinks клиента
     *
     * @param permalinkId - id пермалинка, который заменяем
     * @param newPermalinkId - новый id пермалинка, которым заменяем
     * @param newChainId - новый идентификатор сети
     */
    private fun changePermalinkForCampaignPermalinks(
        shard: Int,
        dslContext: DSLContext,
        clientId: ClientId,
        permalinkId: Long,
        newPermalinkId: Long,
        newChainId: Long?,
    ): Int {
        val campaignIdToChainId = campaignPermalinksOneshotRepository.getCampaignIdToChainId(
            dslContext,
            clientId,
            permalinkId
        )
        if (campaignIdToChainId.isEmpty()) {
            return 0
        }
        logger.info("Change ${campaignIdToChainId.size} campaign_permalinks rows in $shard shard for client = $clientId")

        campaignIdToChainId.forEach {
            logger.info("shard = $shard, clientId = $clientId. Change campaign_permalinks with campaignId = ${it.key}, " +
                "permalink_id from $permalinkId to $newPermalinkId, " +
                "chain_id from ${it.value} to $newChainId")
        }

        return campaignPermalinksOneshotRepository.changePermalinkByCampaignPermalinkIds(
            dslContext,
            campaignIdToChainId.keys,
            permalinkId,
            newPermalinkId,
            newChainId
        )
    }

    /**
     * Меняет пермалинки у banner_permalinks клиента
     *
     * @param permalinkId - id пермалинка, который заменяем
     * @param newPermalinkId - новый id пермалинка, которым заменяем
     * @param newChainId - новый идентификатор сети
     */
    private fun changePermalinkForBannerPermalinks(
        shard: Int,
        dslContext: DSLContext,
        clientId: ClientId,
        permalinkId: Long,
        newPermalinkId: Long,
        newChainId: Long?,
    ): Int {
        var fromBannerId = 0L
        var bannerPermalinksChanged = 0
        while (true) {
            val bannerAndChainIds = bannerPermalinksOneshotRepository.getBannerAndChainIdsByPermalinkId(
                dslContext,
                clientId,
                permalinkId,
                fromBannerId,
                limitBannerPermalink
            )

            bannerAndChainIds.forEach {
                logger.info("shard = $shard, clientId = $clientId. Change banner_permalinks with bannerId = ${it.first}, " +
                    "permalink_id from $permalinkId to $newPermalinkId, " +
                    "chain_id from ${it.second} to $newChainId")
            }

            val bannerIds = bannerAndChainIds.map { it.first }.toSet()
            bannerPermalinksChanged += bannerPermalinksOneshotRepository.changePermalinkByBannerAndPermalinkIds(
                dslContext,
                bannerIds,
                permalinkId,
                newPermalinkId,
                newChainId
            )

            if (bannerAndChainIds.size != limitBannerPermalink) {
                break
            }
            fromBannerId = bannerAndChainIds.maxOf { it.first }
        }

        return bannerPermalinksChanged
    }

    /**
     * Отправляем в телефонию новые данные счетчика и пермалинка
     */
    private fun sendUpdatedDataToTelephony(
        clientId: ClientId,
        clientPhone: ClientPhone,
        newPermalinkId: Long,
        newCounterId: Long,
    ) {
        if (clientPhone.telephonyServiceId == null) {
            return
        }

        val request =
            TelephonyPhoneRequest()
                .withCounterId(newCounterId)
                .withPermalinkId(newPermalinkId)
                .withRedirectPhone(clientPhone.phoneNumber?.phone)
                .withTelephonyServiceId(clientPhone.telephonyServiceId)
        try {
            telephonyClient.linkServiceNumber(clientId.asLong(), request)
        } catch (e: TelephonyClientException) {
            // При ошибке откатываем изменения транзакции
            throw RollbackException()
        }
    }
}
