package ru.yandex.direct.oneshot.oneshots.clean_bid_modifiers_from_conversion_campains

import one.util.streamex.StreamEx
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.oneshots.bsexport.ResyncTableRow
import ru.yandex.direct.oneshot.util.ValidateUtil
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.FunctionalUtils
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtField
import ru.yandex.direct.ytwrapper.model.YtTable

data class CleanParam(
    val ytCluster: YtCluster,
    val tablePath: String,
    val operatorUid: Long
)

data class CleanState(
    val nextRow: Long = 0L
)

val STRATEGY_TYPES_FOR_CLEANING_BID_MODIFIERS = setOf(
    StrategyName.AUTOBUDGET_AVG_CPA,
    StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
    StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER,
    StrategyName.AUTOBUDGET_AVG_CPI,
    StrategyName.AUTOBUDGET_CRR)

val availableCampaignTypes = setOf(
    CampaignType.DYNAMIC,
    CampaignType.MOBILE_CONTENT,
    CampaignType.PERFORMANCE,
    CampaignType.TEXT
)

class ClientsToCleanTableRow : ResyncTableRow<Long>(listOf(CLIENT_ID)) {
    private val clientId: Long?
        get() = valueOf(CLIENT_ID)

    companion object {
        private val CLIENT_ID = YtField("client_id", Long::class.java)
    }

    override fun convert(): Long {
        return clientId ?: throw IllegalStateException("ERROR: found null clientId")
    }

    val id: Long
        get() = clientId ?: throw IllegalStateException("ERROR: found null clientId")
}

@Component
@Approvers("ruslansd", "pavryabov", "kuvshinov", "ssdmitriev", "ninazhevtyak")
@Multilaunch
@Retries(5)
@PausedStatusOnFail
class CleanBidModifiersFromConversionCampaignsOneshot @Autowired constructor(
    private val ytProvider: YtProvider,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val shardHelper: ShardHelper,
    private val bidModifierService: BidModifierService,
    ppcPropertiesSupport: PpcPropertiesSupport
) : SimpleOneshot<CleanParam, CleanState> {
    private val logger = LoggerFactory.getLogger(CleanBidModifiersFromConversionCampaignsOneshot::class.java)

    val chunkSizeProperty = ppcPropertiesSupport.get(PpcPropertyNames.CLEAN_CONVERSION_BID_MODIFIERS_CHUNK_SIZE)

    val relaxTimeProperty = ppcPropertiesSupport.get(PpcPropertyNames.CLEAN_CONVERSION_BID_MODIFIERS_RELAX_TIME)

    override fun validate(inputData: CleanParam): ValidationResult<CleanParam, Defect<*>> {
        val vb = ItemValidationBuilder.of(inputData, Defect::class.java)
        return ValidateUtil.validateTableExistsInYt(ytProvider, vb, inputData.ytCluster, inputData.tablePath)
    }

    override fun execute(inputData: CleanParam, prevState: CleanState?): CleanState? {
        val chunkSize = chunkSizeProperty.getOrDefault(20L)
        val relaxTimeInSeconds = relaxTimeProperty.getOrDefault(200L)

        if (prevState == null) {
            logger.info("First iteration!")
            return CleanState(0)
        }

        val startRow = prevState.nextRow
        val lastRow = startRow + chunkSize

        val requests = getRequestsFromYt(inputData, startRow, lastRow)

        if (requests.isEmpty()) {
            logger.info("Last iteration! startRow: {}, lastRow:{}", startRow, lastRow)
            return null
        }

        logger.info("New iteration! startRow: {}, lastRow:{}", startRow, lastRow)

        //батчом удалять корректировки с нескольких клиентов не умеем, потому в цикле удаляем
        //используем оператор uid супера
        requests.forEach { deleteClientConversionCampaignModifiers(inputData.operatorUid, ClientId.fromLong(it)) }

        logger.info("Iteration finished, sleep for $relaxTimeInSeconds seconds")
        Thread.sleep(relaxTimeInSeconds * 1000)

        return CleanState(lastRow)
    }

    private fun getRequestsFromYt(inputData: CleanParam, startRow: Long, lastRow: Long): List<Long> {
        val ytTable = YtTable(inputData.tablePath)
        val ytOperator = ytProvider.getOperator(inputData.ytCluster)
        val tableRow = createEmptyYtRow()
        val items = mutableListOf<Long>()

        ytOperator.readTableByRowRange(
            ytTable,
            { row -> items.add(row.convert()) },
            tableRow,
            startRow,
            lastRow
        )
        logger.info("Got ${items.size} items from table")
        return items
    }

    private fun createEmptyYtRow() = ClientsToCleanTableRow()

    fun deleteClientConversionCampaignModifiers(operatorUid: Long, clientId: ClientId) {
        val shard: Int = shardHelper.getShardByClientId(clientId)
        val campaignIdsToCleanBidModifiers = getNotArchivedConversionCampaignIds(shard, clientId)
        val campaignsInBatch = 100
        val relaxTimeBetweenCampaignsBatchInSec: Long = 10
        val resultsWithErrors = StreamEx.ofSubLists(campaignIdsToCleanBidModifiers, campaignsInBatch)
            .map {
                logger.info("Start bid modifiers cleaning for cids: " + StreamEx.of(it).joining(","))
                val result = bidModifierService.deleteCampaignModifiers(clientId, operatorUid, it)
                val batchIsFull = it.size == campaignsInBatch
                if (batchIsFull) {
                    //чтобы контролировать нагрузку на транспорт делаем паузу между большими батчами кампаний
                    Thread.sleep(relaxTimeBetweenCampaignsBatchInSec * 1000)
                }
                return@map result
            }
            .remove { it.validationResult.flattenErrors().isEmpty() }
            .toList()
        logger.info("For clientId: " + clientId +
            "has errors on campaign ids: " + StreamEx.of(resultsWithErrors).joining(",") +
            "while remove bid modifiers")
    }

    private fun getNotArchivedConversionCampaignIds(shard: Int, clientId: ClientId): List<Long?>? {

        val campaigns: List<CampaignWithStrategy> = campaignTypedRepository.getSafelyNotArchivedCampaignsWithType(
            shard,
            clientId,
            availableCampaignTypes,
            CampaignWithStrategy::class.java)
        return FunctionalUtils.filterAndMapList(campaigns, { shouldCleanBidModifiers(it) }) { it.id }
    }

    private fun shouldCleanBidModifiers(campaign: CampaignWithStrategy): Boolean {
        return STRATEGY_TYPES_FOR_CLEANING_BID_MODIFIERS.contains(campaign.strategy.strategyName) &&  // Для РМП кампании цель = null когда выбираем 'Установки приложения'
            (StrategyName.AUTOBUDGET_AVG_CPI == campaign.strategy.strategyName || campaign.strategy.strategyData.goalId != null) && //Не удаляем корректировки на мастер кампаниях
            !AvailableCampaignSources.isUC(campaign.source) &&
            campaign.strategy.strategyData.payForConversion != null &&
            campaign.strategy.strategyData.payForConversion
    }
}
