package ru.yandex.direct.oneshot.oneshots.clean_text_fields_for_image_ad_banners;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.oneshot.worker.def.Retries;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

/**
 * Ваншот для очистки полей title, title_extension и body у графических баннеров (banner_type = 'image_ad')
 * Раньше эти поля не использовались и порой заполнялись мусором, типа 'picture banner title'
 * https://st.yandex-team.ru/DIRECT-129366
 * <p>
 * Ваншот принимает пять опциональных параметров:
 * minBid (default: 1)
 * maxBid (default: максимальный bid в таблице)
 * selectSize (default: 10_000_000)
 * updateSize (default: 10_000)
 * maxToUpdate (default: Integer.MAX_VALUE)
 */
@Component
@Multilaunch
@PausedStatusOnFail
@Retries(5)
@Approvers({"saturday"})
@ParametersAreNonnullByDefault
public class CleanTextFieldsForImageAdBannersOneshot implements ShardedOneshot<InputData, State> {
    private static final Logger logger = LoggerFactory.getLogger(CleanTextFieldsForImageAdBannersOneshot.class);
    private static final int DEFAULT_SELECT_SIZE = 10_000_000;
    public static final int DEFAULT_UPDATE_SIZE = 10_000;

    private final AdImageBannerCleanTextFieldsRepository adImageBannerCleanTextFieldsRepository;

    public CleanTextFieldsForImageAdBannersOneshot(
            AdImageBannerCleanTextFieldsRepository adImageBannerCleanTextFieldsRepository) {
        this.adImageBannerCleanTextFieldsRepository = adImageBannerCleanTextFieldsRepository;
    }

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        return ValidationResult.success(inputData);
    }

    @Override
    public State execute(@Nullable InputData inputData, @Nullable State state, int shard) {
        inputData = defaultIfNull(inputData, new InputData());

        int selectSize = defaultIfNull(inputData.getSelectSize(), DEFAULT_SELECT_SIZE);
        int updateSize = defaultIfNull(inputData.getUpdateSize(), DEFAULT_UPDATE_SIZE);
        int maxToUpdate = defaultIfNull(inputData.getMaxToUpdatePerShard(), Integer.MAX_VALUE);
        long maxBid = defaultIfNull(inputData.getMaxBid(), adImageBannerCleanTextFieldsRepository.getMaxBid(shard));

        if (state == null) {
            logger.info("First iteration! input data: {}", inputData);
            sleep();
            long iterMinBid = defaultIfNull(inputData.getMinBid(), 1L);
            long iterMaxBid = Math.min(maxBid, iterMinBid + selectSize - 1);
            return new State(iterMinBid, iterMaxBid, 0);
        }

        long iterMinBid = state.getIterMinBid();
        long iterMaxBid = Math.min(state.getIterMaxBid(), iterMinBid + selectSize - 1);

        if (iterMinBid > iterMaxBid || state.getAlreadyUpdated() >= maxToUpdate) {
            logger.info("Last iteration, input data: {}, state: {}", inputData, state);
            sleep();
            return null;
        } else {
            logger.info("Iteration, input data: {}, state: {}", inputData, state);

            List<BannerWithIds> banners = adImageBannerCleanTextFieldsRepository
                    .getImageAdBannersWithTextFields(shard, iterMinBid, iterMaxBid);

            int updated = state.getAlreadyUpdated();
            for (List<BannerWithIds> batch : Lists.partition(banners, updateSize)) {
                if (updated >= maxToUpdate) {
                    break;
                } else if (batch.size() > maxToUpdate - updated) {
                    batch = batch.subList(0, maxToUpdate - updated);
                }

                adImageBannerCleanTextFieldsRepository.cleanTextFieldsAndBsResync(shard, batch);
                updated += batch.size();
            }

            logger.info("Totally updated on shard: {} banners", updated);
            sleep();
            iterMinBid = iterMaxBid + 1;
            iterMaxBid = Math.min(maxBid, iterMinBid + selectSize - 1);
            return new State(iterMinBid, iterMaxBid, updated);
        }
    }

    private void sleep() {
        try {
            Thread.sleep(3_000);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
