package ru.yandex.direct.oneshot.oneshots.cleanup

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

abstract class BaseCleanupOneshot(
    private val shardHelper: ShardHelper,
) : SimpleOneshot<CleanupParam, Void> {
    companion object {
        const val MAX_IDS_COUNT = 10000
        const val CHUNK_SIZE = 200
    }

    protected val logger: Logger = LoggerFactory.getLogger(this::class.java)

    abstract fun logData(shard: Int, ids: List<Long>)

    abstract fun cleanup(shard: Int, ids: List<Long>)

    final override fun execute(inputData: CleanupParam?, prevState: Void?): Void? {
        inputData!!.ids!!.chunked(CHUNK_SIZE).forEach { chunk ->
            logger.info("Processing next ids chunk")
            logData(inputData.shard!!, chunk)
            cleanup(inputData.shard, chunk)
        }
        return null
    }

    override fun validate(inputData: CleanupParam?): ValidationResult<CleanupParam?, Defect<Any>> {
        val vb = ItemValidationBuilder.of<CleanupParam?, Defect<Any>>(inputData)

        vb.check(CommonConstraints.notNull())
        if (vb.result.hasAnyErrors()) {
            return vb.result
        }

        vb.item(inputData?.shard, "shard")
            .check(CommonConstraints.notNull())
            .check(CommonConstraints.inSet(shardHelper.dbShards().toSet()))

        vb.list(inputData?.ids, "ids")
            .check(CommonConstraints.notNull())
            .check(CollectionConstraints.collectionSize(1, MAX_IDS_COUNT))
            .checkEach(CommonConstraints.validId())

        return vb.result
    }
}
