package ru.yandex.direct.oneshot.oneshots.cleanup

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.repository.BannerRepository
import ru.yandex.direct.core.entity.banner.service.BannerDeleteOperation
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository
import ru.yandex.direct.core.entity.redirectcheckqueue.repository.RedirectCheckQueueRepository
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.Retries
import ru.yandex.direct.operation.Applicability
import ru.yandex.direct.regions.GeoTree
import ru.yandex.direct.regions.Region.GLOBAL_REGION_ID

/**
 * Ваншот для удаления баннеров. Логирует большинство данных перед удалением.
 * На вход нужно передать два параметра {"shard": N, "ids":[]}
 */
@Component
@Multilaunch
@Approvers("kuhtich", "bratgrim", "buhter", "poldnev", "ppalex", "gerdler", "ssdmitriev", "xy6er")
@Retries(3)
class DeleteBanners(
    shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
    private val shardSupport: ShardSupport,
    private val clientGeoService: ClientGeoService,
    private val bannerRepository: BannerRepository,
    private val creativeRepository: CreativeRepository,
    private val turboLandingRepository: TurboLandingRepository,
    private val metrikaCampaignRepository: MetrikaCampaignRepository,
    private val moderationRepository: ModerationRepository,
    private val redirectCheckQueueRepository: RedirectCheckQueueRepository,
) : DeleteBannersDryRun(shardHelper, dslContextProvider) {

    override fun cleanup(shard: Int, ids: List<Long>) {
        logger.info("Going to delete banners: {}", ids)
        BannerDeleteOperationWithoutValidation(shard, ids)
            .executeWithoutValidation()
    }

    inner class BannerDeleteOperationWithoutValidation(
        shard: Int,
        private val bannerIds: List<Long>,
    ) :
        BannerDeleteOperation(
            shard,
            null,   // ClientID, не используется, переопределено через getGeoTreeByClientId
            0, // operator uid, unused
            bannerIds,
            Applicability.PARTIAL,
            dslContextProvider,
            shardSupport,
            null,   // deleteBannerValidationService
            clientGeoService,
            bannerRepository,
            creativeRepository,
            turboLandingRepository,
            metrikaCampaignRepository,
            moderationRepository,
            redirectCheckQueueRepository,
        ) {

        @Override
        override fun getGeoTreeByClientId(clientId: ClientId?): GeoTree {
            return clientGeoService.getClientTranslocalGeoTree(GLOBAL_REGION_ID)
        }

        fun executeWithoutValidation() {
            beforeExecution(bannerIds)
            execute(bannerIds)
            afterExecution(bannerIds)
        }
    }
}
