package ru.yandex.direct.oneshot.oneshots.cleanup

import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag
import ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS
import ru.yandex.direct.dbschema.ppc.tables.MediaplanBanners.MEDIAPLAN_BANNERS
import ru.yandex.direct.dbschema.ppc.tables.Vcards.VCARDS
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.sharding.ShardKey
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.Retries

/**
 * Ваншот для удаления визиток. Логирует перед удалением сами визитки и использовавшие их баннеры.
 * Баннеры отвязывает. org_details и addresses не очищает.
 * На вход нужно передать два параметра {"shard": N, "ids":[]}
 */
@Component
@Multilaunch
@Approvers("kuhtich", "bratgrim", "buhter", "poldnev", "ppalex", "gerdler", "ssdmitriev", "xy6er")
@Retries(3)
class DeleteVcards(
    shardHelper: ShardHelper,
    private val dslContextProvider: DslContextProvider,
    private val shardSupport: ShardSupport,
) : DeleteVcardsDryRun(shardHelper, dslContextProvider) {

    override fun cleanup(shard: Int, ids: List<Long>) {
        val ppcDslContext = dslContextProvider.ppc(shard)

        run {
            val mbids = ppcDslContext.select(MEDIAPLAN_BANNERS.MBID)
                .from(MEDIAPLAN_BANNERS)
                .where(MEDIAPLAN_BANNERS.VCARD_ID.`in`(ids))
                .fetch(MEDIAPLAN_BANNERS.MBID)
            if (mbids.isEmpty()) {
                logger.info("No mediaplan_banners found")
            } else {
                logger.info("Going to remove vcard_id from mediaplan_banners: {}", mbids)
                val mbCnt = ppcDslContext.update(MEDIAPLAN_BANNERS)
                    .setNull(MEDIAPLAN_BANNERS.VCARD_ID)
                    .where(MEDIAPLAN_BANNERS.MBID.`in`(mbids)
                        .and(MEDIAPLAN_BANNERS.VCARD_ID.`in`(ids)))
                    .execute()
                logger.info("Updated {} mediaplan_banners", mbCnt)
            }
        }
        run {
            val bids = ppcDslContext.select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.VCARD_ID.`in`(ids))
                .fetch(BANNERS.BID)
            if (bids.isEmpty()) {
                logger.info("No banners found")
            } else {
                logger.info("Going to remove vcard_id from banners: {}", bids)
                val bCnt = ppcDslContext.update(BANNERS)
                    .setNull(BANNERS.VCARD_ID)
                    // при удалении визитки нужно выставить статус модерации "черновик"
                    .set(BANNERS.PHONEFLAG, BannersPhoneflag.New)
                    .where(BANNERS.BID.`in`(bids)
                        .and(BANNERS.VCARD_ID.`in`(ids)))
                    .execute()
                logger.info("Updated {} banners", bCnt)
            }
        }

        logger.info("Going to delete vcards: {}", ids)
        val vCnt = ppcDslContext.deleteFrom(VCARDS)
            .where(VCARDS.VCARD_ID.`in`(ids))
            .execute()
        logger.info("Deleted {} vcards", vCnt)

        logger.info("Going to delete vcard_id from metabase: {}", ids)
        shardSupport.deleteValues(ShardKey.VCARD_ID, ids)
        logger.info("done")
    }
}
