package ru.yandex.direct.oneshot.oneshots.communication;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.oneshot.base.SimpleYtOneshot;
import ru.yandex.direct.oneshot.base.YtInputData;
import ru.yandex.direct.oneshot.base.YtState;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.communication.CommunicationHelper.sendEventsFromTable;

/**
 * Ваншот для создания событий в платформе коммуникаций по таблице из YTя.
 * Берет строки из таблицы, парсит их как json в события, обновляет некоторые поля и отправкляет
 * https://st.yandex-team.ru/DIRECT-149562
 * <p>
 * Ваншот ожидает два обязательных входных параметра - строку с кластером YT, на котором лежит таблица
 * и строку с путем к таблице с данными
 */
@Component
@Approvers({"a-dubov", "bratgrim"})
@Multilaunch
public class SendCommunicationEventOneshot extends SimpleYtOneshot<YtInputData, YtState> {

    private static final Logger logger = LoggerFactory.getLogger(SendCommunicationEventOneshot.class);
    private static final long CHUNK_SIZE = 100;
    private final CommunicationClient communicationClient;

    public SendCommunicationEventOneshot(
            YtProvider ytProvider,
            CommunicationClient communicationClient) {
        super(ytProvider);
        this.communicationClient = communicationClient;
    }

    @Nullable
    @Override
    public YtState execute(YtInputData inputData, @Nullable YtState prevState) {
        YtCluster ytCluster = YtCluster.parse(inputData.getYtCluster());
        YtTable ytTable = new YtTable(inputData.getTablePath());
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (prevState == null) {
            logger.info("First iteration");
            prevState = new YtState()
                    .withNextRowFromYtTable(0L)
                    .withTotalRowCount(ytOperator.readTableRowCount(ytTable));
        }

        long rowCount = prevState.getTotalRowCount();
        long startRow = prevState.getNextRow();
        long endRow = Math.min(prevState.getNextRow() + CHUNK_SIZE, rowCount);
        if (startRow >= rowCount) {
            logger.info("Last iteration, last processed row: {}, total rows: {}", startRow, rowCount);
            return null;
        }

        int sendMessagesNum = sendEventsFromTable(communicationClient, ytOperator, ytTable, startRow, endRow);
        logger.info("Processed {} events, from row {} to {}", sendMessagesNum, startRow, endRow);

        return new YtState().withNextRowFromYtTable(endRow).withTotalRowCount(rowCount);
    }
}
