package ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.oneshot.base.SimpleOneshotWithoutInput;
import ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.entity.CatalogiaMapRecord;
import ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.entity.CatalogiaTreeRecord;
import ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.repository.CatalogiaMappingYtRepository;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Component
@Approvers({"ammsaid"})
public class ContentCategoriesCatalogiaMapper extends SimpleOneshotWithoutInput {
    private static final Logger logger = LoggerFactory.getLogger(ContentCategoriesCatalogiaMapper.class);

    private static final String CATALOGIA_MAPPING_JSON =
            new ClassPathResourceInputStreamSource("contentcategoriesmapping/catalogia_mapping.json").readText();
    private static final String MAPPING_TABLE_PATH = "export/content_categories/catalogia_categories_mapping";

    private final CatalogiaMappingYtRepository catalogiaMappingYtRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;

    public ContentCategoriesCatalogiaMapper(
            CatalogiaMappingYtRepository catalogiaMappingYtRepository,
            CryptaSegmentRepository cryptaSegmentRepository
    ) {
        this.catalogiaMappingYtRepository = catalogiaMappingYtRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
    }

    @Override
    public void execute() {
        var catalogiaCategories = catalogiaMappingYtRepository.readTree();
        var map = fromJson(CATALOGIA_MAPPING_JSON, CatalogiaMapRecord[].class);
        var childrenCats = fillCatalogiaChildrenCats(map, catalogiaCategories);

        var goalsToIdMap = cryptaSegmentRepository.getBrandSafetyAndContentSegments();

        var parentMapping = getCategoriesMapping(goalsToIdMap, childrenCats);
        try {
            catalogiaMappingYtRepository.saveMapping(parentMapping, MAPPING_TABLE_PATH);
        } catch (SQLException e) {
            logger.error("Can't save mapping", e);
        }
    }

    private HashMap<String, HashSet<Long>> getCategoriesMapping(
            Map<Long, Goal> goalsToIdMap, HashMap<Long, HashSet<Long>> mapping) {
        var allMapping = new HashMap<String, HashSet<Long>>();

        for (var goal : goalsToIdMap.values()) {
            if (goal.getParentId() == null || goal.getParentId() == 0) {
                // map parent as is
                appendCatsToMapping(allMapping, goal.getKeywordValue(), mapping.get(goal.getId()));
            } else {
                var parent = goalsToIdMap.get(goal.getParentId());

                // if it's "other" child and it has siblings map parent here excluding other children
                var siblings = getDirectCategoryChildren(goal.getParentId(), goalsToIdMap.values());
                if ("Остальное".equals(goal.getName()) && siblings.size() > 1) {
                    if (parent != null && mapping.get(parent.getId()) != null) {
                        // map parent category to "other"
                        var catalogiaDirectIds = new HashSet<>(mapping.get(parent.getId()));
                        // exclude all siblings mapping from "other"
                        for (var sibling : siblings) {
                            var siblingCats = mapping.get(sibling.getId());
                            if (siblingCats != null) {
                                catalogiaDirectIds.removeAll(siblingCats);
                            }
                        }

                        appendCatsToMapping(allMapping, goal.getKeywordValue(), catalogiaDirectIds);
                    }
                } else {
                    // else it's a real child, so map it as is
                    appendCatsToMapping(allMapping, goal.getKeywordValue(), mapping.get(goal.getId()));
                }

                // map all children to parent
                if (parent != null) {
                    appendCatsToMapping(allMapping, parent.getKeywordValue(), mapping.get(goal.getId()));
                }
            }
        }

        return allMapping;
    }

    private List<Goal> getDirectCategoryChildren(Long goalId, Collection<Goal> goals) {
        return goals.stream().filter(g -> goalId.equals(g.getParentId())).collect(toList());
    }

    private static void appendCatsToMapping(
            HashMap<String, HashSet<Long>> mapping, String keyword, HashSet<Long> catalogiaDirectIds) {
        if (catalogiaDirectIds == null) {
            return;
        }

        if (mapping.containsKey(keyword)) {
            mapping.get(keyword).addAll(catalogiaDirectIds);
        } else {
            mapping.put(keyword, catalogiaDirectIds);
        }
    }

    private static HashMap<Long, HashSet<Long>> fillCatalogiaChildrenCats(
            CatalogiaMapRecord[] map, List<CatalogiaTreeRecord> catalogiaCats) {
        var categoryNameToIdMap = listToMap(
                catalogiaCats, CatalogiaTreeRecord::getCategoryName, CatalogiaTreeRecord::getDirectId);

        // map goalId to set of catalogia ids
        var categoryMapping = new HashMap<Long, HashSet<Long>>();
        for (var record : map) {
            var mappedCats = new ArrayList<Long>();

            // collect all children of mapped record, except "minus" ones
            for (var categoryName : record.getPlusCats()) {
                if (categoryName.contains("v")) {
                    // map all virtuals with categoryName tag except minus ones

                    var allMinusCatIds = new ArrayList<>();
                    for (var minusName : record.getMinusCats()) {
                        var minusDirectId = categoryNameToIdMap.get(minusName);
                        allMinusCatIds.addAll(
                                collectChildrenCats(minusDirectId, catalogiaCats, emptyList(), categoryNameToIdMap));
                    }

                    var allMinusCatNames = allMinusCatIds.stream()
                            .map(id -> catalogiaCats.stream()
                                    .filter(cat -> id.equals(cat.getDirectId()))
                                    .map(CatalogiaTreeRecord::getCategoryName)
                                    .findAny().orElse(null))
                            .filter(Objects::nonNull)
                            .collect(toList());

                    var virtuals = filterAndMapList(
                            catalogiaCats,
                            c -> {
                                boolean isMinus = allMinusCatNames.stream()
                                        .filter(minusName -> c.getCategoryName().contains(minusName))
                                        .findAny()
                                        .orElse(null) != null;

                                return c.getCategoryName().contains(categoryName) && !isMinus;
                            },
                            CatalogiaTreeRecord::getDirectId);

                    mappedCats.addAll(virtuals);
                } else {
                    var categoryDirectId = categoryNameToIdMap.get(categoryName);
                    if (categoryDirectId != null) {
                        mappedCats.add(categoryDirectId);
                        var childrenCats = collectChildrenCats(
                                categoryDirectId, catalogiaCats, record.getMinusCats(), categoryNameToIdMap);
                        mappedCats.addAll(childrenCats);
                    }
                }
            }

            categoryMapping.put(record.getGoalId(), new HashSet<>(mappedCats));
        }

        return categoryMapping;
    }

    private static List<Long> excludeMinusCats(
            List<Long> children, List<String> minusCats, Map<String, Long> categoryNameToIdMap) {
        var minusCatsIds = minusCats.stream()
                .map(categoryNameToIdMap::get)
                .filter(Objects::nonNull)
                .collect(toList());

        return children.stream()
                .filter(child -> !minusCatsIds.contains(child))
                .collect(toList());
    }

    private static ArrayList<Long> collectChildrenCats(
            Long categoryDirectId,
            List<CatalogiaTreeRecord> catalogiaCats,
            List<String> minusCats,
            Map<String, Long> categoryNameToIdMap) {
        var children = catalogiaCats.stream()
                // do not need to map virtuals here
                .filter(c -> categoryDirectId.equals(c.getDirectParentId()) && !c.getCategoryName().contains("v"))
                .map(CatalogiaTreeRecord::getDirectId)
                .collect(toList());

        children = excludeMinusCats(children, minusCats, categoryNameToIdMap);

        var allChildren = new ArrayList<>(children);
        for (var child : children) {
            allChildren.addAll(collectChildrenCats(child, catalogiaCats, minusCats, categoryNameToIdMap));
        }

        return allChildren;
    }
}
