package ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.repository;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Repository;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.entity.CatalogiaTreeRecord;
import ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.entity.ytmodels.generated.YtCatalogiaTreeRow;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;

import static ru.yandex.direct.oneshot.oneshots.contentcategoriesmapping.entity.ytmodels.generated.YtDbTables.CATALOGIATREE;

@Repository
@ParametersAreNonnullByDefault
public class CatalogiaMappingYtRepository {
    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final String YQL_TEMPLATE =
            "INSERT INTO `%s` WITH TRUNCATE (keyword_value, catalogia_id) VALUES %s";
    private static final int NUMBER_OF_PARAMETERS = 2;

    private final YtProvider ytProvider;

    public CatalogiaMappingYtRepository(YtProvider ytProvider) {
        this.ytProvider = ytProvider;
    }

    public List<CatalogiaTreeRecord> readTree() {
        var ytOperator = ytProvider.getOperator(YT_CLUSTER, YtSQLSyntaxVersion.SQLv1);
        return ytOperator.readTableAndMap(
                CATALOGIATREE, new YtCatalogiaTreeRow(), CatalogiaMappingYtRepository::convertFromYt);
    }

    private static CatalogiaTreeRecord convertFromYt(YtCatalogiaTreeRow row) {
        return new CatalogiaTreeRecord()
                .withCategoryId(row.getCategoryID())
                .withDirectId(row.getDirectID())
                .withDirectParentId(row.getDirectParentID());
    }


    public void saveMapping(Map<String, HashSet<Long>> mapping, String path) throws SQLException {
        var ytClusterConfig = ytProvider.getClusterConfig(YT_CLUSTER);
        var ytPath = YtPathUtil.generatePath(ytClusterConfig.getHome(), relativePart(), path);

        var records = flattenMapping(mapping);

        var yqlDataSource = ytProvider.getYql(YT_CLUSTER, YtSQLSyntaxVersion.SQLv1);
        try (var connection = yqlDataSource.getConnection()) {
            var repeat = "(?,?),".repeat(records.size());
            var questionMarks = repeat.substring(0, repeat.length() - 1);
            var yql = String.format(YQL_TEMPLATE, ytPath, questionMarks);

            try (var statement = connection.prepareStatement(yql)) {
                for (var i = 0; i < records.size(); i++) {
                    statement.setString(i * NUMBER_OF_PARAMETERS + 1, records.get(i).get1());
                    statement.setLong(i * NUMBER_OF_PARAMETERS + 2, records.get(i).get2());
                }

                statement.execute();
            }

        }
    }

    private ArrayList<Tuple2<String, Long>> flattenMapping(Map<String, HashSet<Long>> mapping) {
        var result = new ArrayList<Tuple2<String, Long>>();
        for (var directKeywordValue : mapping.keySet()) {
            for (var catalogiaId : mapping.get(directKeywordValue)) {
                result.add(new Tuple2<>(directKeywordValue, catalogiaId));
            }
        }

        return result;
    }

    private static String relativePart() {
        EnvironmentType environmentType = Environment.getCached();
        if (environmentType.isProductionOrPrestable()) {
            return "./";
        } else {
            return String.format("%s_%s", System.getProperty("user.name"), environmentType.name().toLowerCase());
        }
    }
}
