package ru.yandex.direct.oneshot.oneshots.create_global_ab_segments;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignExperiment;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignWithBrandLiftExperimentsService;
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter;
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientFactory;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.retargeting.model.ExperimentRetargetingConditions;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

@Component
@Multilaunch
@Approvers({"andreypav", "eboguslavskaya"})
public class CreateGlobalABSegmentsCpmOneshot implements ShardedOneshot<Void, Void> {

    private static final Logger logger = LoggerFactory.getLogger(CreateGlobalABSegmentsCpmOneshot.class);

    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final ClientService clientService;
    private final CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService;
    private final ShardHelper shardHelper;
    private final RetargetingConditionService retargetingConditionService;
    private final RequestBasedMetrikaClientFactory requestBasedMetrikaClientFactory;
    private final DslContextProvider dslContextProvider;
    private final CampaignModifyRepository campaignModifyRepository;

    public CreateGlobalABSegmentsCpmOneshot(CampaignRepository campaignRepository,
                                            CampaignTypedRepository campaignTypedRepository,
                                            ClientService clientService,
                                            CampaignWithBrandLiftExperimentsService campaignWithBrandLiftExperimentsService,
                                            ShardHelper shardHelper, RetargetingConditionService retargetingConditionService,
                                            RequestBasedMetrikaClientFactory requestBasedMetrikaClientFactory,
                                            DslContextProvider dslContextProvider, CampaignModifyRepository campaignModifyRepository) {
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.clientService = clientService;
        this.campaignWithBrandLiftExperimentsService = campaignWithBrandLiftExperimentsService;
        this.shardHelper = shardHelper;
        this.retargetingConditionService = retargetingConditionService;
        this.requestBasedMetrikaClientFactory = requestBasedMetrikaClientFactory;
        this.dslContextProvider = dslContextProvider;
        this.campaignModifyRepository = campaignModifyRepository;
    }


    @Nullable
    @Override
    public Void execute(Void inputData, Void prevState, int shard) {
        List<Long> cpmCampaignsWithoutExperiment = campaignRepository.getCpmCampaignIdsWithoutExperiment(shard);
        List<CampaignWithBrandLift> campaignWithBrandLifts =
                StreamEx.of(campaignTypedRepository.getTypedCampaigns(shard, cpmCampaignsWithoutExperiment))
                        .select(CampaignWithBrandLift.class).toList();
        var cidsByClientId = StreamEx.of(campaignWithBrandLifts)
                .mapToEntry(CommonCampaign::getClientId, c -> c)
                .mapKeys(ClientId::fromLong)
                .grouping();
        logger.info("Setting global AB experiment for clientIds: {} on shard {}", cidsByClientId.keySet(), shard);
        List<AppliedChanges<CampaignWithBrandLift>> appliedChanges =
                EntryStream.of(cidsByClientId)
                        .mapKeyValue(this::setExperimentForCampaign)
                        .flatMap(Collection::stream)
                        .collect(Collectors.toList());
        var dslContext = dslContextProvider.ppc(shard);
        if (!appliedChanges.isEmpty()) {
            logger.info("Setting global AB experiment for campaigns with ids: {} on shard {}", cpmCampaignsWithoutExperiment, shard);
            campaignModifyRepository.updateCampaignsTable(dslContext, appliedChanges);
        }
        return null;
    }

    private List<AppliedChanges<CampaignWithBrandLift>> setExperimentForCampaign(ClientId clientId, List<CampaignWithBrandLift> campaigns) {
        String login = shardHelper.getLoginByUid(clientService.getClient(clientId).getChiefUid());

        var preparedExperimentsGlobal = StreamEx.of(campaigns)
                .flatMapToEntry(model -> campaignWithBrandLiftExperimentsService.prepareExperiment(login, model))
                .distinctValues()
                .grouping();

        List<CampaignExperiment> campaignExperiments = campaigns.stream()
                .map(c -> new CampaignExperiment()
                        .withAbSegmentGoalIds(c.getAbSegmentGoalIds())
                        .withSectionIds(c.getSectionIds())).distinct().collect(Collectors.toList());

        RequestBasedMetrikaClientAdapter metrikaClient =
                requestBasedMetrikaClientFactory.createMetrikaClient(clientId, campaigns);
        List<ExperimentRetargetingConditions> retConditions =
                retargetingConditionService.findOrCreateExperimentsRetargetingConditions(
                        clientId,
                        campaignExperiments,
                        metrikaClient.getAbSegmentGoals(),
                        preparedExperimentsGlobal);
        if (retConditions.size() != 1) {
            logger.info("Couldn't create ret_conditions for client with clientID: {}", clientId.asLong());
            return null;
        }

        var retCondition = retConditions.get(0);
        return campaigns.stream().map(camp ->
                new ModelChanges<>(camp.getId(), CampaignWithBrandLift.class)
                    .process(retCondition.getRetargetingConditionId(),
                            CampaignWithBrandLift.AB_SEGMENT_RETARGETING_CONDITION_ID)
                    .process(retCondition.getStatisticRetargetingConditionId(),
                            CampaignWithBrandLift.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID)
                    .process(Boolean.TRUE, CampaignWithBrandLift.IS_CPM_GLOBAL_AB_SEGMENT)
                    .process(CampaignStatusBsSynced.NO, CampaignWithBrandLift.STATUS_BS_SYNCED)
                    .applyTo(camp)
        ).collect(Collectors.toList());
    }

    @Override
    public ValidationResult<Void, Defect> validate(Void inputData) {
        return ValidationResult.success(inputData);
    }
}
