package ru.yandex.direct.oneshot.oneshots.creativeupdatepresetid;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.oneshot.oneshots.creativeupdatepresetid.repository.MongoVideoAddtionsRepository;
import ru.yandex.direct.oneshot.oneshots.creativeupdatepresetid.repository.PerfCreativesRepository;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;


/**
 * https://st.yandex-team.ru/DIRECT-130551
 * Входные данные: список creativeIds, preset_id
 * Меняет preset_id и layout_id переданных креативов на указанный
 */
@Component
@Multilaunch
@Approvers({"ivatkoegor"})
public class CreativeUpdatePresetIdOneshot implements SimpleOneshot<CreativeUpdatePresetIdOneshotData, Void> {

    private static final Logger logger = LoggerFactory.getLogger(CreativeUpdatePresetIdOneshot.class);

    private final ShardHelper shardHelper;
    private final MongoVideoAddtionsRepository mongoVideoAddtionsRepository;
    private final PerfCreativesRepository perfCreativesRepository;

    @Autowired
    public CreativeUpdatePresetIdOneshot(
            ShardHelper shardHelper,
            MongoVideoAddtionsRepository mongoVideoAddtionsRepository,
            PerfCreativesRepository perfCreativesRepository
    ) {
        this.shardHelper = shardHelper;
        this.mongoVideoAddtionsRepository = mongoVideoAddtionsRepository;
        this.perfCreativesRepository = perfCreativesRepository;
    }

    @Override
    public Void execute(CreativeUpdatePresetIdOneshotData inputData, Void state) {
        Long updatedCreativesCount = mongoVideoAddtionsRepository.updatePresetId(inputData.getCreativeIds(),
                inputData.getPresetId());
        logger.info("The number of updated creatives in mongo {}", updatedCreativesCount);

        shardHelper.forEachShard(shard -> {
            int updatedRecordsCount = perfCreativesRepository.updateLayoutId(shard, inputData.getCreativeIds(),
                    inputData.getPresetId());
            logger.info("The number of updated records at shard {}: {}", shard, updatedRecordsCount);
        });
        return null;
    }

    @Override
    public ValidationResult<CreativeUpdatePresetIdOneshotData, Defect> validate(CreativeUpdatePresetIdOneshotData inputData) {
        ItemValidationBuilder<CreativeUpdatePresetIdOneshotData, Defect> vb = ItemValidationBuilder.of(inputData);
        vb.list(inputData.getCreativeIds(), "creativeIds")
                .check(notNull())
                .check(notEmptyCollection())
                .checkEach(notNull())
                .checkEach(validId());
        vb.item(inputData.getPresetId(), "presetId")
                .check(notNull())
                .check(validId());

        return vb.getResult();
    }
}
