package ru.yandex.direct.oneshot.oneshots.crr_last_bidder_restart_time

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.oneshots.crr_last_bidder_restart_time.CrrStrategyLastBidderRestartTimeOneshot.Companion.CrrStrategySetLastBidderRestartTimePosition
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.time.LocalDateTime

/**
 * Ваншот для выставления дефолтного lastBidderRestartTime в CRR стратегиях.
 * https://st.yandex-team.ru/DIRECT-146816
 */

@Component
@Multilaunch
@PausedStatusOnFail
@Approvers("ruslansd", "ssdmitriev")
class CrrStrategyLastBidderRestartTimeOneshot(
    private val campaignTypedRepository: CampaignTypedRepository,
    private val campaignModifyRepository: CampaignModifyRepository
) : ShardedOneshot<Void, CrrStrategySetLastBidderRestartTimePosition> {

    private val logger = LoggerFactory.getLogger(CrrStrategyLastBidderRestartTimeOneshot::class.java)

    override fun validate(inputData: Void?): ValidationResult<Void, Defect<Any>> =
        ValidationResult.success(inputData)

    override fun execute(inputData: Void?, prevPosition: CrrStrategySetLastBidderRestartTimePosition?, shard: Int): CrrStrategySetLastBidderRestartTimePosition? {
        val position = prevPosition ?: CrrStrategySetLastBidderRestartTimePosition(0L)
        logger.info("Start iteration: shard [$shard], last campaignId [${position.lastCampaignId}]")

        return when (val result = oneIteration(shard, position)) {
            Finished -> {
                logger.info("Last iteration on shard [$shard]")
                null
            }
            is NextPosition -> {
                logger.info("Successfully finish iteration: shard [$shard], last campaignId [${result.position.lastCampaignId}]")
                result.position
            }
        }
    }

    private fun oneIteration(shard: Int, position: CrrStrategySetLastBidderRestartTimePosition): IterationResult {
        val campaigns = campaignTypedRepository.getCampaignsForTypesAndIdGreaterThan(
            shard,
            position.lastCampaignId,
            CRR_STRATEGY_CAMPAIGN_TYPES,
            LIMIT
        )
        val updatedCrrStrategies = campaigns.mapNotNull { it as? CampaignWithStrategy }
            .filter { isCrrStrategy(it) && hasEmptyLastBidderRestartTime(it) }
            .mapNotNull { setLastBidderRestartTime(it) }

        if (updatedCrrStrategies.isNotEmpty()) {
            campaignModifyRepository.updateCampaignsTable(shard, updatedCrrStrategies)
        }

        val nextPosition = campaigns.maxByOrNull { it.id }?.let { NextPosition(CrrStrategySetLastBidderRestartTimePosition(it.id)) }

        return nextPosition ?: Finished
    }

    companion object {

        private val CRR_STRATEGY_CAMPAIGN_TYPES = setOf(CampaignsType.dynamic, CampaignsType.performance, CampaignsType.text)

        data class CrrStrategySetLastBidderRestartTimePosition(val lastCampaignId: Long)

        sealed interface IterationResult

        object Finished : IterationResult

        data class NextPosition(val position: CrrStrategySetLastBidderRestartTimePosition) : IterationResult

        private fun hasEmptyLastBidderRestartTime(campaign: CampaignWithStrategy): Boolean =
            campaign.strategy?.strategyData?.lastBidderRestartTime == null

        private fun isCrrStrategy(campaign: CampaignWithStrategy): Boolean =
            campaign.strategy?.strategyName == StrategyName.AUTOBUDGET_CRR

        private fun setLastBidderRestartTime(campaign: CampaignWithStrategy) =
            newWithLastBidderRestartTime(campaign.strategy)?.let { updatedStrategy ->
                ModelChanges(campaign.id, CampaignWithStrategy::class.java)
                    .process(updatedStrategy, CampaignWithStrategy.STRATEGY)
                    .applyTo(campaign)
            }

        private fun newWithLastBidderRestartTime(strategy: DbStrategy?): DbStrategy? =
            strategy?.let { CampaignStrategyUtils.getStrategyCopyWithLasBidderRestartTime(it, START_OF_CRR_STRATEGY_DATE) }

        private const val LIMIT = 10000

        //Дата анонса ДРР стратегии.
        private val START_OF_CRR_STRATEGY_DATE = LocalDateTime.parse("2021-06-02T00:00:00")
    }

}
