package ru.yandex.direct.oneshot.oneshots.delete_keywords_from_bids_base

import com.google.common.collect.Lists
import org.jooq.exception.DataAccessException
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.common.util.RelaxedWorker.UnsafeCallable
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.oneshot.oneshots.delete_keywords_from_bids_base.repository.OneshotBidsBaseRepository
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

data class State(
    val fromBidId: Long,
    val deletedCount: Long = 0
)

data class InputData(
    val clientId: Long?
)

/**
 * Удаляет все keywords из bids_base таблицы
 *
 * После каждой итерации делается пауза, которая регулируется пропертей DELETE_KEYWORDS_FROM_BIDS_BASE_RELAX_TIME
 * Так же при каждом удалении чанка bids делается relax
 */
@Component
@Multilaunch
@Approvers("mexicano", "ppalex", "ajkon", "zhur")
class DeleteKeywordsFromBidsBaseOneshot : ShardedOneshot<InputData?, State?> {

    companion object {
        private val logger = LoggerFactory.getLogger(DeleteKeywordsFromBidsBaseOneshot::class.java)
        private const val DELETE_CHUNK_SIZE_BID_IDS = 1_000
        private const val DELETE_SLEEP_COEF = 1.0
        private const val DEFAULT_RELAX_TIME_BETWEEN_ITERATIONS = 1L
        private const val MAX_REQUEST_ATTEMPTS = 5
    }

    private val oneshotBidsBaseRepository: OneshotBidsBaseRepository
    private val ppcPropertiesSupport: PpcPropertiesSupport
    private val selectChunkSizeBidIds: Long
    private val shardHelper: ShardHelper

    @Autowired
    constructor(oneshotBidsBaseRepository: OneshotBidsBaseRepository,
                ppcPropertiesSupport: PpcPropertiesSupport,
                shardHelper: ShardHelper) {
        this.oneshotBidsBaseRepository = oneshotBidsBaseRepository
        this.ppcPropertiesSupport = ppcPropertiesSupport
        this.shardHelper = shardHelper
        this.selectChunkSizeBidIds = 20_000L
    }

    /**
     * Используется для тестов
     */
    constructor(oneshotBidsBaseRepository: OneshotBidsBaseRepository,
                ppcPropertiesSupport: PpcPropertiesSupport,
                selectChunkSizeBidIds: Long,
                shardHelper: ShardHelper) {
        this.oneshotBidsBaseRepository = oneshotBidsBaseRepository
        this.ppcPropertiesSupport = ppcPropertiesSupport
        this.selectChunkSizeBidIds = selectChunkSizeBidIds
        this.shardHelper = shardHelper
    }

    override fun execute(
        inputData: InputData?,
        prevState: State?,
        shard: Int
    ): State? {
        val state = prevState ?: State(0)

        var attemptsLeft = MAX_REQUEST_ATTEMPTS;

        while (attemptsLeft > 0) {
            try {
                val relaxTimeProperty = ppcPropertiesSupport.get(PpcPropertyNames.DELETE_KEYWORDS_FROM_BIDS_BASE_RELAX_TIME)
                val relaxTimeBetweenIterations = relaxTimeProperty.getOrDefault(DEFAULT_RELAX_TIME_BETWEEN_ITERATIONS)

                return run(shard, inputData?.clientId, relaxTimeBetweenIterations, state)
            } catch (e: DataAccessException) {
                if (--attemptsLeft <= 0) {
                    throw e
                }
                logger.error("Couldn't connect to database. Attempts left $attemptsLeft", e)
                try {
                    Thread.sleep(5000L)
                } catch (interruptedException: InterruptedException) {
                    Thread.currentThread().interrupt()
                    throw RuntimeException("Thread was interrupted", e)
                }
            }
        }
        return null
    }

    private fun run(
        shard: Int,
        clientId: Long?,
        relaxTimeBetweenIterations: Long,
        state: State,
    ): State? {
        val keywordBids = if (clientId != null) {
            oneshotBidsBaseRepository
                .getKeywordsByIdsForClient(shard, clientId, state.fromBidId, selectChunkSizeBidIds)
        } else {
            oneshotBidsBaseRepository.getKeywordsByIds(shard, state.fromBidId, selectChunkSizeBidIds)
        }
        val countDeleted = deleteKeywordsFromBidsBase(shard, keywordBids)

        return if (keywordBids.size < selectChunkSizeBidIds) {
            logger.info("work completed! total deleted {}", state.deletedCount + countDeleted)
            null
        } else {
            logger.info("got {} keywords from bids_base from {} to {} ids, deleted {}",
                keywordBids.size, keywordBids.first(), keywordBids.last(), countDeleted)
            logger.info("shard=$shard: iteration finished, sleep for $relaxTimeBetweenIterations seconds")
            Thread.sleep(relaxTimeBetweenIterations * 1000)
            State(keywordBids.last(), state.deletedCount + countDeleted)
        }
    }

    private fun deleteKeywordsFromBidsBase(shard: Int, bids: List<Long>): Int {
        var countDeleted = 0
        val relaxed = RelaxedWorker(DELETE_SLEEP_COEF)
        for (deleteChunk in Lists.partition(bids, DELETE_CHUNK_SIZE_BID_IDS)) {
            countDeleted += relaxed.callAndRelax(UnsafeCallable<Int, RuntimeException> {
                oneshotBidsBaseRepository.deleteKeywordsByIds(shard, deleteChunk)
            })
        }
        return countDeleted
    }

    override fun validate(inputData: InputData?): ValidationResult<InputData?, Defect<*>> {

        if (inputData?.clientId == null) {
            return ValidationResult.success(inputData)
        }

        val builder: ItemValidationBuilder<InputData, Defect<*>> = ItemValidationBuilder.of(inputData)
        builder.item(inputData.clientId, "clientId")
            .check(CommonConstraints.notNull())
            .check(NumberConstraints.greaterThan(0L))
            .check(
                Constraint.fromPredicate(
                    { clientId -> isClientExists(clientId) }, CommonDefects.objectNotFound()
                ), When.isValid()
            )
        return builder.result
    }

    private fun isClientExists(clientId: Long): Boolean {
        return shardHelper.isExistentClientId(clientId)
    }
}
