package ru.yandex.direct.oneshot.oneshots.ecomcatalog

import java.util.Comparator.comparing
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.model.EcomOfferCatalog
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomOfferCatalogsRepository
import ru.yandex.direct.gemini.GeminiClient
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.CollectionConstraints.unique
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan
import ru.yandex.direct.validation.constraint.StringConstraints.validUrl
import ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject

data class CatalogInfo(
    val url: String,
    val title: String?,
    val catalogPath: String?,
    val imageUrl: String?,
    val visitsCount: Long,
    val isPermanent: Boolean
)

data class CatalogsSendingParam(
    val catalogs: List<CatalogInfo>
)

private val LOGGER = LoggerFactory.getLogger(AddEcomOfferCatalogsOneshot::class.java)

private const val MAX_CATALOGS_SIZE = 5000
private const val CHUNK_SIZE = 1000

/**
 * Oneshot для добавления вручную каталогов в табличку `ecom_offer_catalogs`.
 * Если по введенному урлу уже существует каталог - обновляет его.
 */
@Component
@Multilaunch
@Approvers("kozobrodov", "buhter", "dmitanosh")
class AddEcomOfferCatalogsOneshot(
    private val ecomOfferCatalogsRepository: EcomOfferCatalogsRepository,
    private val geminiClient: GeminiClient
) : SimpleOneshot<CatalogsSendingParam, Void?> {

    override fun validate(inputData: CatalogsSendingParam): ValidationResult<CatalogsSendingParam, Defect<*>> {
        return validateObject(inputData) {
            listProperty(CatalogsSendingParam::catalogs) {
                check(notNull())
                check(notEmptyCollection(), When.isValid())
                check(maxListSize(MAX_CATALOGS_SIZE), When.isValid())

                checkEach(unique(comparing { it.url }), duplicatedObject(), When.isValid())
                checkEachBy(this@AddEcomOfferCatalogsOneshot::validateCatalogInfo, When.isValid())
            }
        }
    }

    private fun validateCatalogInfo(catalogInfo: CatalogInfo): ValidationResult<CatalogInfo, Defect<*>> {
        return validateObject(catalogInfo) {
            property(CatalogInfo::url) {
                check(validUrl())
            }
            property(CatalogInfo::visitsCount) {
                check(notLessThan(0L))
            }
        }
    }

    override fun execute(inputData: CatalogsSendingParam, prevState: Void?): Void? {
        LOGGER.info("START")

        val catalogsByHost = inputData.catalogs
            .groupBy { getHost(it.url) }

        LOGGER.info("Processing ${inputData.catalogs.size} catalogs by ${catalogsByHost.size} hosts")

        val mainMirrors = geminiClient.getMainMirrors(catalogsByHost.keys)

        val existingCatalogsByHosts = ecomOfferCatalogsRepository.getByHosts(mainMirrors.values)
            .groupBy { it.host }

        val catalogsToUpdate = mutableListOf<AppliedChanges<EcomOfferCatalog>>()
        val catalogsToAdd = mutableListOf<EcomOfferCatalog>()

        catalogsByHost.forEach { (host, catalogs) ->
            val mainMirror = mainMirrors[host]
            if (mainMirror != null) {
                val existingCatalogs = existingCatalogsByHosts[mainMirror]
                    ?.associateBy { it.url }
                    ?: emptyMap()

                val (existingCatalogsInfo, newCatalogsInfo) =
                    catalogs.partition { existingCatalogs.containsKey(it.url) }

                existingCatalogsInfo.map { toEcomOfferCatalogChanges(it, existingCatalogs[it.url]!!) }
                    .also { catalogsToUpdate.addAll(it) }

                newCatalogsInfo.map { toEcomOfferCatalog(it, mainMirror) }
                    .also { catalogsToAdd.addAll(it) }
            } else {
                LOGGER.error("Main mirror for urls $catalogs not found")
            }
        }

        updateCatalogs(catalogsToUpdate)
        addCatalogs(catalogsToAdd)

        LOGGER.info("FINISH")
        return null
    }

    private fun updateCatalogs(catalogsToUpdate: List<AppliedChanges<EcomOfferCatalog>>) {
        if (catalogsToUpdate.isEmpty()) {
            LOGGER.info("No catalogs to update")
            return
        }

        LOGGER.info("Updating ${catalogsToUpdate.size} catalogs")
        val updated = catalogsToUpdate.chunked(CHUNK_SIZE)
            .map { ecomOfferCatalogsRepository.update(it) }
            .sum()
        LOGGER.info("Updated $updated catalogs")
    }

    private fun addCatalogs(catalogsToAdd: List<EcomOfferCatalog>) {
        if (catalogsToAdd.isEmpty()) {
            LOGGER.info("No catalogs to add")
            return
        }

        LOGGER.info("Inserting ${catalogsToAdd.size} catalogs")
        val inserted = catalogsToAdd.chunked(CHUNK_SIZE)
            .map { ecomOfferCatalogsRepository.insert(it) }
            .sum()
        LOGGER.info("Inserted $inserted catalogs")
    }

    private fun toEcomOfferCatalog(catalogInfo: CatalogInfo, host: String): EcomOfferCatalog {
        return EcomOfferCatalog().apply {
            this.host = host
            title = catalogInfo.title
            catalogPath = catalogInfo.catalogPath
            url = catalogInfo.url
            imageUrl = catalogInfo.imageUrl
            visitsCount = catalogInfo.visitsCount
            isPermanent = catalogInfo.isPermanent
        }
    }

    private fun toEcomOfferCatalogChanges(
        catalogInfo: CatalogInfo,
        existingCatalog: EcomOfferCatalog
    ): AppliedChanges<EcomOfferCatalog> {
        return ModelChanges(existingCatalog.id, EcomOfferCatalog::class.java)
            .process(catalogInfo.title, EcomOfferCatalog.TITLE)
            .process(catalogInfo.catalogPath, EcomOfferCatalog.CATALOG_PATH)
            .process(catalogInfo.imageUrl, EcomOfferCatalog.IMAGE_URL)
            .process(catalogInfo.visitsCount, EcomOfferCatalog.VISITS_COUNT)
            .process(catalogInfo.isPermanent, EcomOfferCatalog.IS_PERMANENT)
            .applyTo(existingCatalog)
    }

    private fun getHost(url: String): String {
        val urlParts = UrlParts.fromUrl(url)
        return "${urlParts.protocol}://${urlParts.domain}"
    }
}
