package ru.yandex.direct.oneshot.oneshots.featurestickets;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueRef;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.ThreadUtils.sleep;

@Service
public class CreateFeaturesTicketsOneshotStartrekService {

    private static final Logger logger = LoggerFactory.getLogger(CreateFeaturesTicketsOneshotStartrekService.class);

    private static final String QUEUE = "DIRECT";
    private static final String TICKET_TYPE = "task";
    private static final String COMPONENT_NAME = "FeatureFlag";
    private static final String SUMMARY_TEMPLATE = "Feature: %s";
    private static final String CAN_NOT_CHECK_TICKET_EXISTENCE = "CAN_NOT_CHECK_TICKET_EXISTENCE";

    private final Session session;

    @Autowired
    public CreateFeaturesTicketsOneshotStartrekService(DirectConfig directConfig, EnvironmentType environmentType) {

        String oauthToken = "";
        try {
            oauthToken = FileUtils.slurp(FileUtils.expandHome(directConfig.getString(
                    "startrek.robot_dt_oneshots.token_file"))).trim();
        } catch (RuntimeException e) {
            logger.warn("Can't read token for @robot-dt-oneshots", e);
        }

        // используем тестовый стартрек, если окружение не продовое
        String apiUrl;
        if (environmentType.equals(EnvironmentType.PRODUCTION)) {
            DirectConfig startrekConfig = directConfig.getBranch("startrek");
            apiUrl = startrekConfig.getString("startrek_api_url");
        } else {
            apiUrl = "https://st-api.test.yandex-team.ru";
        }

        this.session = StartrekClientBuilder.newBuilder()
                .uri(apiUrl)
                .maxConnections(10)
                .build(oauthToken);
    }

    /**
     * Для каждой фичи из features создать тикет в стартреке.
     * Если тикет уже существует, то новый тикет не будет создан.
     *
     * @return мапа featureName => ticketId (только для новых тикетов)
     */
    public Map<String, String> createTickets(Collection<Feature> features) {
        ComponentRef componentRef = getComponentRef();

        Map<String, String> result = new HashMap<>();
        features.forEach(feature -> {
            String ticketId = createTicket(feature, componentRef);
            if (ticketId != null) {
                result.put(feature.getFeatureTextId(), ticketId);
            }
        });
        return result;
    }

    /**
     * Создает тикет и возвращает ticketId.
     * Если тикет уже существует или произошла ошибка при создании, то ticketId не возвращается.
     */
    @Nullable
    private String createTicket(Feature feature, ComponentRef componentRef) {
        String featureName = feature.getFeatureTextId();

        String ticketKey = getTicketKey(featureName);
        if (ticketKey != null) {
            if (CAN_NOT_CHECK_TICKET_EXISTENCE.equals(ticketKey)) {
                logger.info("Couldn't check existence of the ticket for feature: {}", featureName);
            } else {
                logger.info("Issue for feature {} already exists: {}", featureName, ticketKey);
            }
            return null;
        }

        Set<String> watchers = feature.getSettings().getOriginalWatchers();
        IssueCreate issueCreate = IssueCreate.builder()
                .queue(QUEUE)
                .type(TICKET_TYPE)
                .summary(String.format(SUMMARY_TEMPLATE, featureName))
                .description("//Тикет создан с помощью " + CreateFeaturesTicketsOneshot.class.getSimpleName() + "//")
                .components(componentRef)
                .assignee(feature.getSettings().getOriginalOwner())
                .followers(watchers == null ? null : watchers.toArray(new String[0]))
                .build();

        String errorMessage =
                "Exception occurred during creating ticket for oneshot " + CreateFeaturesTicketsOneshot.class;

        ticketKey =
                new RetryCommand<String>()
                        .withErrorMessage(errorMessage)
                        .run(() -> session.issues().create(issueCreate).getKey());

        logger.info("Issue for feature {} created: {}", featureName, ticketKey);
        return ticketKey;
    }

    /**
     * Возвращает:
     * null, если тикет не существует
     * ticketId, если тикет существует
     * CAN_NOT_CHECK_TICKET_EXISTENCE, если не удалось определить существует тикет или нет
     */
    private String getTicketKey(String featureName) {
        String query =
                String.format("\"Queue\": \"%s\" " +
                                "AND \"Summary\": \"Feature: %s\" " +
                                "AND \"Components\": \"%s\"",
                        QUEUE, featureName, COMPONENT_NAME);

        String errorMessage = "Exception occurred during get ticket for oneshot " + CreateFeaturesTicketsOneshot.class;
        String expectedSummary = String.format(SUMMARY_TEMPLATE, featureName);

        List<String> issueKeyList =
                new RetryCommand<List<String>>()
                        .withDefaultValue(List.of(CAN_NOT_CHECK_TICKET_EXISTENCE))
                        .withErrorMessage(errorMessage)
                        .run(() ->
                                session.issues().find(query).stream()
                                        .filter(issue -> expectedSummary.equals(issue.getSummary()))
                                        .map(IssueRef::getKey)
                                        .collect(toList())
                        );

        if (issueKeyList.size() > 1) {
            logger.warn("Multiple tickets with same key {}, search result: {}", featureName, issueKeyList);
        }

        return issueKeyList.isEmpty() ? null : issueKeyList.get(0);
    }

    private ComponentRef getComponentRef() {
        return session.queues().get(QUEUE).getComponents().stream()
                .filter(c -> COMPONENT_NAME.equalsIgnoreCase(c.getDisplay()))
                .findAny()
                .orElseThrow(() -> new RuntimeException(
                        "Can't get ComponentRef for oneshot " + CreateFeaturesTicketsOneshot.class));
    }


    private static class RetryCommand<T> {
        private static final int RETRIES = 5;
        private static final long DELAY = 500L;
        private String errorMessage;
        private T defaultValue;

        public RetryCommand<T> withDefaultValue(T defaultValue) {
            this.defaultValue = defaultValue;
            return this;
        }

        public RetryCommand<T> withErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        private T run(Supplier<T> supplier) {
            RuntimeException lastException = null;
            for (int i = 0; i < RETRIES; ++i) {
                try {
                    return supplier.get();
                } catch (RuntimeException e) {
                    lastException = e;
                    sleep(DELAY);
                }
            }

            logger.error("{} \"{}\"", errorMessage, lastException);
            return defaultValue;
        }
    }

}
