package ru.yandex.direct.oneshot.oneshots.feed

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feed.feedWithUsageTypeToAppliedChanges
import ru.yandex.direct.core.entity.feed.model.FeedUsageType
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedUsageService
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import ru.yandex.direct.oneshot.worker.def.ShardedOneshot
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import java.util.EnumSet

data class FeedUpdateUsageTypesParam(
    val feedIds: List<Long>?,
    val minFeedId: Long?
)

data class FeedUpdateUsageTypesState(
    val lastFeedId: Long,
    val chunkNumber: Int
)

/**
 * НАЗВАНИЮ ВАНШОТА НЕ ВЕРИТЬ!
 * Oneshot выбирает фиды, которые Директ синкнуты с MBI, пересчитывает и сохраняет в базу их использование,
 * Может запускаться на всех фидах или по списку feedIds
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "zakhar")
class DropFeedUsageTypeOneshot(
    private val feedRepository: FeedRepository,
    private val feedUsageService: FeedUsageService
) : ShardedOneshot<FeedUpdateUsageTypesParam, FeedUpdateUsageTypesState> {
    companion object {
        private val logger = LoggerFactory.getLogger(DropFeedUsageTypeOneshot::class.java)
        private const val CHUNK_SIZE = 100
    }

    override fun validate(inputData: FeedUpdateUsageTypesParam?): ValidationResult<FeedUpdateUsageTypesParam, Defect<Any>> {
        return ValidationResult.success(inputData)
    }

    override fun execute(
        inputData: FeedUpdateUsageTypesParam?,
        prevState: FeedUpdateUsageTypesState?,
        shard: Int
    ): FeedUpdateUsageTypesState? {
        val chunkNumber = prevState?.chunkNumber ?: 1
        if (chunkNumber == 1) {
            logger.info("START on shard $shard chunk number $chunkNumber")
        } else {
            logger.info("Start chunk number $chunkNumber on shard $shard")
        }
        if (inputData?.feedIds != null) {
            logger.info("Requested total ${inputData.feedIds.size} feeds to recalculate feed usage")
        }
        val minFeedId = prevState?.lastFeedId ?: inputData?.minFeedId ?: 0

        val feedIds = feedRepository.getFeedIdsWithMarketShopId(shard, inputData?.feedIds, minFeedId, CHUNK_SIZE)

        if (feedIds.isEmpty()) {
            logger.info("FINISH on shard $shard chunk number $chunkNumber is empty")
            return null
        }

        val actualFeedUsageTypeByFeedId = feedUsageService.getActualFeedUsageTypeByFeedId(shard, feedIds)
        updateFeedUsageType(shard, actualFeedUsageTypeByFeedId)

        val lastFeedId = feedIds.last()
        if (feedIds.size < CHUNK_SIZE) {
            logger.info("FINISH on shard $shard chunk number $chunkNumber with last feed id $lastFeedId.")
            return null
        }

        logger.info("Finish chunk number $chunkNumber  on shard $shard with last feed id $lastFeedId")

        return FeedUpdateUsageTypesState(lastFeedId, chunkNumber + 1)
    }

    private fun updateFeedUsageType(shard: Int, feedUsageTypeByFeedIdToProcess: Map<Long, EnumSet<FeedUsageType>>) {
        val feedsByFeedId = feedRepository.getSimple(shard, feedUsageTypeByFeedIdToProcess.keys).associateBy { it.id }
        val feedsToUpdateUsage = feedUsageTypeByFeedIdToProcess.mapKeys { feedsByFeedId[it.key]!! }
        val appliedChanges = feedWithUsageTypeToAppliedChanges(feedsToUpdateUsage)

        feedRepository.update(shard, appliedChanges)
    }
}
