package ru.yandex.direct.oneshot.oneshots.feed

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.dbschema.ppc.Tables.FEEDS
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.oneshot.base.SimpleOneshotWithoutInput
import ru.yandex.direct.oneshot.worker.def.Approvers
import ru.yandex.direct.oneshot.worker.def.Multilaunch
import java.time.LocalDateTime

/**
 * Oneshot проставляет фидам, которые были зарегистированы в MBI и не используются в активных кампаниях
 * значение last_used, чтобы джоба смогла выключить из в MBI, чтобы Datacamp мог удалить оффер этих фидов,
 * чтобы освободить ресурсов
 * last_used проставляется размазанным во времени(по 20 фидов на 5 минуту), чтобы не было слишком
 * массового отключения в MBI про работе джобы DisableFeedsInMBIJob
 * Multilaunch — на всякий случай, по идее одного запуска должно быть достаточно, за актуальностью следит
 */
@Component
@Multilaunch
@Approvers("buhter", "kozobrodov", "zakhar")
class SetLastUsedForOutdatedFeedsOneshot @Autowired constructor(
    private val shardHelper: ShardHelper,
    private val feedRepository: FeedRepository
) : SimpleOneshotWithoutInput() {
    companion object {
        private val logger = LoggerFactory.getLogger(SetLastUsedForOutdatedFeedsOneshot::class.java)
        private const val UPDATE_CHUNK_SIZE = 100
    }

    override fun execute() {
        logger.info("START")

        val feedsByShard = shardHelper.forEachShardSequential {
            return@forEachShardSequential feedRepository.getSimple(
                it,
                FEEDS.MARKET_FEED_ID.isNotNull
                    .and(FEEDS.LAST_USED.isNull)
                    .and(
                        FEEDS.USAGE_TYPE.isNull
                            .or(FEEDS.USAGE_TYPE.eq(""))
                    )
            )
        }
        val now = LocalDateTime.now()
        var counter = 0
        feedsByShard.forEach { (shard, feeds) ->
            logger.info("Start with ${feeds.size} feeds on shard: $shard")

            val ac = feeds.filterNotNull()
                .map { feed ->
                    ModelChanges(feed.id, Feed::class.java)
                        .process(now.plusSeconds(((++counter / 20) * 60).toLong()), Feed.LAST_USED)
                        .applyTo(feed as Feed)
                }
            logger.info("Updating ${ac.size} records in DB for shard: $shard")
            ac.chunked(UPDATE_CHUNK_SIZE).forEach { feedRepository.update(shard, it) }
        }
        logger.info("FINISH")
    }

}
