package ru.yandex.direct.oneshot.oneshots.flatcpcstrategyfromyt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jooq.exception.DataAccessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.oneshot.base.ShardedYtOneshot;
import ru.yandex.direct.oneshot.base.YtInputData;
import ru.yandex.direct.oneshot.base.YtState;
import ru.yandex.direct.oneshot.oneshots.flatcpcstrategyfromyt.entity.ytmodels.generated.YtFlatCPCBidsByCidRow;
import ru.yandex.direct.oneshot.oneshots.flatcpcstrategyfromyt.service.FlatCpcStrategyFromYtMigrationOneshotService;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.PausedStatusOnFail;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;

/**
 * Ваншот заполняет ставки на сеть на кампании, которые представлены в табличке:
 * https://yt.yandex-team.ru/hahn/navigation?path=//home/direct/tmp/m-troitskiy/flat_cpc_bids_tables/FlatCPCBidsByCids
 * Если для кампании не все ставки находятся в этой табличке, то для них ставки на сеть выставляются равными ставкам
 * на поиск
 * <p>
 * Алгоритм ваншота:
 * - достаём из таблички шарды, id кампаний, а также новые ставки на неё
 * - достаём для id кампаний ставки из таблицы bids_base, выставляем ставки на сеть равными ставкам на поиск
 * - достаём для id кампаний ставки из таблицы bids, для тех что нашлись выставляем новую ставку, для тех что не
 * нашлись, ставку равную поиску
 * - выставляем кампаниям тип стратегии на different_places, выставляем context_limit = 0 и enable_cpc_hold = false
 */

@Component
@Multilaunch
@PausedStatusOnFail
@Approvers({"ssdmitriev", "pavelkataykin"})
public class FlatCpcStrategyFromYtMigrationOneshot extends ShardedYtOneshot<YtInputData, YtState> {

    private static final Logger logger = LoggerFactory.getLogger(FlatCpcStrategyFromYtMigrationOneshot.class);
    private static final long CHUNK_SIZE = 1000;
    private static final long DEFAULT_RELAX_TIME_BETWEEN_ITERATIONS = 1;
    private static final int MAX_REQUEST_ATTEMPTS = 5;

    private final FlatCpcStrategyFromYtMigrationOneshotService oneshotService;
    private final PpcProperty<Long> relaxTimeProperty;

    @Autowired
    protected FlatCpcStrategyFromYtMigrationOneshot(
            YtProvider ytProvider,
            FlatCpcStrategyFromYtMigrationOneshotService oneshotService,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        super(ytProvider);
        this.oneshotService = oneshotService;
        relaxTimeProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.FLAT_CPC_FROM_YT_MIGRATION_RELAX_TIME);
    }


    @Nullable
    @Override
    public YtState execute(YtInputData inputData, YtState prevState, int shard) {
        YtCluster ytCluster = YtCluster.parse(inputData.getYtCluster());
        YtTable ytTable = new YtTable(inputData.getTablePath());
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);

        if (prevState == null) {
            logger.info("First iteration, shard {}", shard);
            prevState = new YtState()
                    .withNextRowFromYtTable(0L)
                    .withTotalRowCount(ytOperator.readTableRowCount(ytTable));
        }

        long rowCount = prevState.getTotalRowCount();
        long startRow = prevState.getNextRow();
        long endRow = Math.min(prevState.getNextRow() + CHUNK_SIZE, rowCount);

        if (startRow >= rowCount) {
            logger.info("Last iteration, shard: {}, last processed row: {}, total rows: {}", shard, startRow, rowCount);
            return null;
        }

        //Читаем данные из YT
        Map<Long, String> newBidsFromYtByCid = getNewBidsFromYtByCid(
                shard,
                ytTable,
                ytOperator,
                startRow,
                endRow
        );

        long relaxTimeBetweenIterations = relaxTimeProperty.getOrDefault(DEFAULT_RELAX_TIME_BETWEEN_ITERATIONS);
        run(newBidsFromYtByCid, relaxTimeBetweenIterations, shard);

        return new YtState().withNextRowFromYtTable(endRow).withTotalRowCount(rowCount);
    }

    @NotNull
    private Map<Long, String> getNewBidsFromYtByCid(int shard, YtTable ytTable, YtOperator ytOperator, long startRow,
                                                    long endRow) {
        int attemptsLeft = MAX_REQUEST_ATTEMPTS;
        Map<Long, String> newBidsFromYtByCid = new HashMap<>();
        while (attemptsLeft > 0) {
            try {
                YtFlatCPCBidsByCidRow tableRow = new YtFlatCPCBidsByCidRow();
                // Заполняем мапу cid --> новые ставки взятые из YT в строковом формате
                ytOperator.readTableByRowRange(
                        ytTable,
                        getYtFlatCPCBidsByCidRowConsumer(shard, newBidsFromYtByCid),
                        tableRow,
                        startRow,
                        endRow
                );
                return newBidsFromYtByCid;
            } catch (DataAccessException e) {
                if (--attemptsLeft <= 0) {
                    throw e;
                }
                logger.error("Couldn't connect to database. Attempts left " + attemptsLeft, e);
                try {
                    Thread.sleep(5000L);
                } catch (InterruptedException interruptedException) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("Thread was interrupted", e);
                }
            }
        }
        return newBidsFromYtByCid;
    }

    @NotNull
    private Consumer<YtFlatCPCBidsByCidRow> getYtFlatCPCBidsByCidRowConsumer(int shard,
                                                                             Map<Long, String> newBidsFromYt) {
        return row -> {
            if (shard != row.getShard()) {
                return;
            }

            String bidsFromYtSting = row.getBids();
            Long cid = row.getCid();
            newBidsFromYt.put(cid, bidsFromYtSting);
        };
    }

    @Nullable
    public void run(Map<Long, String> newBidsFromYtByCid, long relaxTimeBetweenIterations, int shard) {
        //Берем cids кампаний из YT
        Set<Long> cidsFromYt = newBidsFromYtByCid.keySet();

        //Достаём из ppc кампании для которых достали ставки из YT
        List<CampaignWithCustomStrategy> flatCpcCampaigns = oneshotService.getFlatCpcCampaigns(shard, cidsFromYt);

        //Чтобы ваншот не падал при применении изменений для кампаний если список пуст
        if (flatCpcCampaigns.isEmpty()) {
            return;
        }

        //Берём cid кампаний которые собираемся изменять
        List<Long> cids = StreamEx.of(flatCpcCampaigns)
                .map(CampaignWithCustomStrategy::getId)
                .toList();

        //Достаём keywords, выставляем price_context для тех что нашлись в ставках из yt, для тех что не нашлись
        // выставляем равным price
        Long keywordsCount = oneshotService.setPriceContextForBidsInKeywordsForCids(shard, newBidsFromYtByCid, cids);

        //Достаем ставки из bids_base, выставляем price_context равный price, сохраняем
        int bidsFromBidsBaseCount = oneshotService.setPriceContextInBidsFromBidsBase(shard, cids);

        //Найдём архивные кампании, для них достанем архивные ставки и выставим price_context из таблички, или
        // равный price, если ставки в табличке нет
        List<Long> archivedCids = StreamEx.of(flatCpcCampaigns)
                .select(CommonCampaign.class)
                .filter(CommonCampaign::getStatusArchived)
                .map(CommonCampaign::getId)
                .toList();

        int archivedKeywordsCount = oneshotService.setPriceContextInBidsFromBidsArc(shard, newBidsFromYtByCid,
                archivedCids);

        //Делаем изменения для стратегии, enable_cpc_hold и context_limit
        oneshotService.setChangesForCampaigns(shard, flatCpcCampaigns);

        logger.info("Iteration finished, updated campaigns: {}, bids in bids: {}, " +
                        "bids in bids_base: {}, bids in bids_arc: {}",
                cids.size(), keywordsCount, bidsFromBidsBaseCount, archivedKeywordsCount);
        logger.info("Iteration finished on shard: {}, sleep for {} seconds", shard, relaxTimeBetweenIterations);

        sleep(relaxTimeBetweenIterations);
    }

    private void sleep(Long sleepTime) {
        try {
            Thread.sleep(sleepTime * 1000);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
