package ru.yandex.direct.oneshot.oneshots.flatcpcstrategyfromyt.service;


import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.DynamicCampaign;
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.TextCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.repository.KeywordMapping;
import ru.yandex.direct.core.entity.keyword.repository.KeywordRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.oneshots.flatcpcstrategyfromyt.entity.BidFromYt;
import ru.yandex.direct.oneshot.oneshots.flatcpcstrategyfromyt.repository.FlatCpcFromYtMigrationOneshotBidsArcRepository;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class FlatCpcStrategyFromYtMigrationOneshotService {

    private static final Logger logger = LoggerFactory.getLogger(FlatCpcStrategyFromYtMigrationOneshotService.class);
    private static final int BIDS_LOG_BATCH_SIZE = 1000;
    private static final int BIDS_ARC_UPDATE_BATCH_SIZE = 500;

    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;
    private final KeywordRepository keywordRepository;
    private final BidRepository bidRepository;
    private final FlatCpcFromYtMigrationOneshotBidsArcRepository bidsArcRepository;

    private final PpcProperty<Boolean> shouldLogBidsChangesProperty;

    @Autowired
    public FlatCpcStrategyFromYtMigrationOneshotService(
            CampaignTypedRepository campaignTypedRepository,
            CampaignModifyRepository campaignModifyRepository,
            KeywordRepository keywordRepository,
            BidRepository bidRepository,
            FlatCpcFromYtMigrationOneshotBidsArcRepository bidsArcRepository,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
        this.keywordRepository = keywordRepository;
        this.bidRepository = bidRepository;
        this.bidsArcRepository = bidsArcRepository;
        this.shouldLogBidsChangesProperty =
                ppcPropertiesSupport.get(PpcPropertyNames.FLAT_CPC_FROM_YT_MIGRATION_SHOULD_LOG_BIDS_CHANGES);
    }

    //Методы для чтения/записи кампаний
    public List<CampaignWithCustomStrategy> getFlatCpcCampaigns(int shard, Collection<Long> cids) {
        List<? extends BaseCampaign> campaigns = campaignTypedRepository.getTypedCampaigns(shard, cids);

        return StreamEx.of(campaigns)
                .select(CampaignWithCustomStrategy.class)
                .filter(campaign -> campaign.getStrategy().getPlatform() == CampaignsPlatform.BOTH
                        && campaign.getStrategy().getStrategyName() == StrategyName.DEFAULT_
                        && campaign.getStrategy().getStrategy() != CampOptionsStrategy.DIFFERENT_PLACES)
                .toList();
    }

    public void setChangesForCampaigns(int shard, List<CampaignWithCustomStrategy> flatCpcCampaigns) {

        List<TextCampaign> textCampaigns = StreamEx.of(flatCpcCampaigns)
                .select(TextCampaign.class)
                .toList();
        List<MobileContentCampaign> mobileContentCampaigns = StreamEx.of(flatCpcCampaigns)
                .select(MobileContentCampaign.class)
                .toList();
        List<DynamicCampaign> dynamicCampaigns = StreamEx.of(flatCpcCampaigns)
                .select(DynamicCampaign.class)
                .toList();

        if (!textCampaigns.isEmpty()) {
            List<AppliedChanges<TextCampaign>> textCampaignChanges = new ArrayList<>();
            textCampaigns.forEach(campaign -> fillChangesForTextCampaign(textCampaignChanges, campaign));
            campaignModifyRepository.updateCampaignsTable(shard, textCampaignChanges);

            logger.info("Updated text campaigns to different_places strategy with ids: {}", mapList(textCampaignChanges,
                    changes -> changes.getNewValue(TextCampaign.ID)));
        }

        if (!mobileContentCampaigns.isEmpty()) {
            List<AppliedChanges<MobileContentCampaign>> mobileContentCampaignChanges = new ArrayList<>();
            mobileContentCampaigns.forEach(campaign -> fillChangesForMobileContentCampaign(mobileContentCampaignChanges, campaign));
            campaignModifyRepository.updateCampaignsTable(shard, mobileContentCampaignChanges);

            logger.info("Updated mobile content campaigns to different_places strategy with ids: {}",
                    mapList(mobileContentCampaignChanges,
                            changes -> changes.getNewValue(MobileContentCampaign.ID)));
        }

        if (!dynamicCampaigns.isEmpty()) {
            List<AppliedChanges<DynamicCampaign>> dynamicCampaignChanges = new ArrayList<>();
            dynamicCampaigns.forEach(campaign -> fillChangesForDynamicCampaign(dynamicCampaignChanges, campaign));
            campaignModifyRepository.updateCampaignsTable(shard, dynamicCampaignChanges);

            logger.info("Updated dynamic campaigns to different_places strategy with ids: {}",
                    mapList(dynamicCampaignChanges,
                            changes -> changes.getNewValue(DynamicCampaign.ID)));
        }
    }

    private void fillChangesForTextCampaign(List<AppliedChanges<TextCampaign>> textCampaignChanges,
                                            TextCampaign campaign) {
        DbStrategy newStrategy = campaign.getStrategy().copy();
        newStrategy.setStrategy(CampOptionsStrategy.DIFFERENT_PLACES);
        textCampaignChanges.add(new ModelChanges<>(campaign.getId(), TextCampaign.class)
                .process(newStrategy, TextCampaign.STRATEGY)
                .process(false, TextCampaign.ENABLE_CPC_HOLD)
                .process(0, TextCampaign.CONTEXT_LIMIT)
                .applyTo(campaign));
    }

    private void fillChangesForMobileContentCampaign(List<AppliedChanges<MobileContentCampaign>> mobileContentCampaignChanges,
                                                     MobileContentCampaign campaign) {
        DbStrategy newStrategy = campaign.getStrategy().copy();
        newStrategy.setStrategy(CampOptionsStrategy.DIFFERENT_PLACES);
        mobileContentCampaignChanges.add(new ModelChanges<>(campaign.getId(), MobileContentCampaign.class)
                .process(newStrategy, MobileContentCampaign.STRATEGY)
                .process(false, MobileContentCampaign.ENABLE_CPC_HOLD)
                .process(0, MobileContentCampaign.CONTEXT_LIMIT)
                .applyTo(campaign));
    }

    private void fillChangesForDynamicCampaign(List<AppliedChanges<DynamicCampaign>> dynamicCampaignChanges,
                                               DynamicCampaign campaign) {
        DbStrategy newStrategy = campaign.getStrategy().copy();
        newStrategy.setStrategy(CampOptionsStrategy.DIFFERENT_PLACES);
        dynamicCampaignChanges.add(new ModelChanges<>(campaign.getId(), DynamicCampaign.class)
                .process(newStrategy, DynamicCampaign.STRATEGY)
                .process(false, DynamicCampaign.ENABLE_CPC_HOLD)
                .applyTo(campaign));
    }

    //Методы для изменения price_context в bids_base
    public int setPriceContextInBidsFromBidsBase(int shard, List<Long> cids) {
        List<Bid> oldBids = bidRepository.getBidsWithRelevanceMatchByCampaignIds(shard, cids);
        List<AppliedChanges<Bid>> bidsChanges = new ArrayList<>();

        oldBids.forEach(bid -> bidsChanges.add(new ModelChanges<>(bid.getId(), Bid.class)
                .process(bid.getPrice(), Bid.PRICE_CONTEXT)
                .applyTo(bid))
        );

        int result = bidRepository.setBidsInBidsBase(shard, bidsChanges);
        if (shouldLogBidsChangesProperty.getOrDefault(true)) {
            StreamEx.ofSubLists(bidsChanges, BIDS_LOG_BATCH_SIZE)
                    .forEach(batch -> logger.info("Updated bids in bids_base: {}", mapList(batch,
                            FlatCpcStrategyFromYtMigrationOneshotService::buildLogMessageForBidChanges)));
        }
        return result;
    }

    //Методы для изменения price_context в bids
    public Long setPriceContextForBidsInKeywordsForCids(
            int shard,
            Map<Long, String> newBidsFromYt,
            List<Long> cids) {
        long changedCount = 0L;
        for (Long cid : cids) {
            List<Keyword> oldKeywords = keywordRepository.getKeywordsByCampaignId(shard, cid);
            List<BidFromYt> newKeywordBids = getBidsFromYtString(newBidsFromYt.get(cid));

            List<AppliedChanges<Keyword>> keywordsChanges = new ArrayList<>();
            oldKeywords.forEach(keyword -> fillKeywordChangesFromKeyword(newKeywordBids, keywordsChanges, keyword));

            changedCount += bidRepository.setBidsInBids(shard, keywordsChanges);

            if (shouldLogBidsChangesProperty.getOrDefault(true)) {
                StreamEx.ofSubLists(keywordsChanges, BIDS_LOG_BATCH_SIZE)
                        .forEach(batch -> logger.info("Updated bid in bids: {}", mapList(keywordsChanges,
                                FlatCpcStrategyFromYtMigrationOneshotService::buildLogMessageForKeywordChanges)));
            }
        }
        return changedCount;
    }

    //Здесь мы ищем для keyword подходящую ставку из ставок из таблички YT по связке phraseId + adGroupId и добавляем
    // изменения в keywordChanges
    //Если не находим, выставляем priceContext такой же, как и price
    private static void fillKeywordChangesFromKeyword(
            List<BidFromYt> newKeywordBids,
            List<AppliedChanges<Keyword>> keywordsChanges,
            @NotNull Keyword keyword
    ) {
        BigInteger phraseId = keyword.getPhraseBsId();
        Long adGroupId = keyword.getAdGroupId();
        Optional<BidFromYt> newBid = findBidFromNewKeywordBids(newKeywordBids, phraseId, adGroupId);
        getChangesForOptionalBid(keywordsChanges, keyword, newBid);
    }

    private static Optional<BidFromYt> findBidFromNewKeywordBids(List<BidFromYt> newKeywordBids,
                                                                 BigInteger phraseId, Long adGroupId) {
        return StreamEx.of(newKeywordBids)
                .filter(k -> k.getPhraseId().equals(phraseId))
                .filter(k -> k.getAdGroupID().equals(adGroupId))
                .findFirst();
    }

    private static void getChangesForOptionalBid(List<AppliedChanges<Keyword>> keywordChanges, Keyword keyword,
                                                 Optional<BidFromYt> newBid) {
        if (newBid.isPresent()) {
            keywordChanges.add(new ModelChanges<>(keyword.getId(), Keyword.class)
                    .process(BigDecimal.valueOf(newBid.get().getPriceContext()), Keyword.PRICE_CONTEXT)
                    .applyTo(keyword));
        } else {
            keywordChanges.add(new ModelChanges<>(keyword.getId(), Keyword.class)
                    .process(KeywordMapping.priceToDbFormat(keyword.getPrice()), Keyword.PRICE_CONTEXT)
                    .applyTo(keyword));
        }
    }

    //Методы для изменения price_context в bids_arc
    public int setPriceContextInBidsFromBidsArc(
            int shard,
            Map<Long, String> newBidsFromYtByCid,
            List<Long> archivedCids) {
        Map<Long, List<Keyword>> oldArchivedKeywords = keywordRepository.getArchivedKeywordsByAdGroupIds(shard, null,
                archivedCids, Collections.emptySet());

        Map<Long, List<BidFromYt>> bidsFromYtByAdGroupId = EntryStream.of(newBidsFromYtByCid)
                .mapValues(FlatCpcStrategyFromYtMigrationOneshotService::getBidsFromYtString)
                .values()
                .flatMap(List::stream)
                .groupingBy(BidFromYt::getAdGroupID);

        List<Keyword> newKeywordList = EntryStream.of(oldArchivedKeywords)
                .map(entry -> setPriceContextInBidsArcForKeyword(shard, bidsFromYtByAdGroupId, entry.getKey(),
                        entry.getValue()))
                .flatMap(List::stream)
                .toList();

        StreamEx.ofSubLists(newKeywordList, BIDS_ARC_UPDATE_BATCH_SIZE)
                .forEach(batch -> updateAndLogArchivedKeywords(shard, newKeywordList, batch));

        return newKeywordList.size();
    }

    //Парсинг ставок из yt в формате json
    private static List<BidFromYt> getBidsFromYtString(String bidsJsonString) {
        ObjectMapper objectMapper = new ObjectMapper();
        TypeFactory typeFactory = objectMapper.getTypeFactory();
        List<BidFromYt> bidsFromYt = new ArrayList<>();
        try {
            bidsFromYt = objectMapper.readValue(bidsJsonString, typeFactory.constructCollectionType(List.class,
                    BidFromYt.class));
        } catch (IOException e) {
            logger.error("Cannot parse bids from YT");
            throw new RuntimeException("Cannot parse bids from YT", e);
        }
        return bidsFromYt;
    }

    private List<Keyword> setPriceContextInBidsArcForKeyword(int shard,
                                                             Map<Long, List<BidFromYt>> newBidsFromYtByAdGroupIds,
                                                             Long adGroupId,
                                                             List<Keyword> keywords) {
        List<Keyword> newKeywordList = new ArrayList<>();
        for (Keyword keyword : keywords) {
            BigInteger phraseBsId = keyword.getPhraseBsId();
            Optional<BidFromYt> newArchivedBid = StreamEx.of(newBidsFromYtByAdGroupIds.getOrDefault(adGroupId,
                            List.of()))
                    .filter(bidFromYt -> bidFromYt.getPhraseId().equals(phraseBsId))
                    .findFirst();
            if (newArchivedBid.isPresent()) {
                keyword.setPriceContext(BigDecimal.valueOf(newArchivedBid.get().getPriceContext()));
            } else {
                keyword.setPriceContext(KeywordMapping.priceToDbFormat(keyword.getPrice()));
            }
            newKeywordList.add(keyword);
        }

        return newKeywordList;
    }

    private void updateAndLogArchivedKeywords(int shard, List<Keyword> newKeywordList, List<Keyword> batch) {
        bidsArcRepository.updatePriceContextInBidsArc(shard, batch);
        logger.info("Updated bids in bids_arc: {}", mapList(newKeywordList,
                FlatCpcStrategyFromYtMigrationOneshotService::buildLogMessageForArchivedBids));
    }

    private static String buildLogMessageForBidChanges(AppliedChanges<Bid> changes) {
        return "id : " + changes.getNewValue(Bid.ID)
                + ", cid: " + changes.getNewValue(Bid.CAMPAIGN_ID)
                + ", price: " + changes.getNewValue(Bid.PRICE)
                + ", " + changes;
    }

    private static String buildLogMessageForKeywordChanges(AppliedChanges<Keyword> changes) {
        return "id : " + changes.getNewValue(Keyword.ID)
                + ", cid: " + changes.getNewValue(Keyword.CAMPAIGN_ID)
                + ", price: " + changes.getNewValue(Keyword.PRICE)
                + ", " + changes;
    }

    private static String buildLogMessageForArchivedBids(Keyword newArchivedKeyword) {
        return "id: " + newArchivedKeyword.getId()
                + ", cid: " + newArchivedKeyword.getCampaignId()
                + ", pid: " + newArchivedKeyword.getPhraseBsId()
                + ", price: " + newArchivedKeyword.getPrice()
                + ", price_context: " + newArchivedKeyword.getPriceContext();
    }
}
