package ru.yandex.direct.oneshot.oneshots.generatesmartcreatives;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.ListenableFuture;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.bannerstorage.client.BannerStorageClientConfiguration;
import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbschema.ppc.enums.UsersRepType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.bannerstorage.AddCreativeResult;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.bannerstorage.AddCreativesResult;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.bannerstorage.CreativeGroupDTO;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.bannerstorage.GroupIdDTO;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.bannerstorage.GroupIdsDTO;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.intapi.SpawnCreativeResult;
import ru.yandex.direct.oneshot.oneshots.generatesmartcreatives.intapi.SpawnCreativesResult;
import ru.yandex.direct.oneshot.worker.def.Approvers;
import ru.yandex.direct.oneshot.worker.def.Multilaunch;
import ru.yandex.direct.oneshot.worker.def.SimpleOneshot;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.http.HttpHeaders.ACCEPT;
import static org.apache.http.HttpHeaders.AUTHORIZATION;
import static org.apache.http.HttpHeaders.CONTENT_TYPE;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppcdict.tables.ShardClientId.SHARD_CLIENT_ID;
import static ru.yandex.direct.dbschema.ppcdict.tables.ShardCreativeId.SHARD_CREATIVE_ID;
import static ru.yandex.direct.tvm.TvmIntegration.SERVICE_TICKET_HEADER;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.validation.Predicates.in;
import static ru.yandex.direct.validation.Predicates.notEmptyCollection;
import static ru.yandex.direct.validation.Predicates.notNull;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;

/**
 * Выполняет дозаполнение смартовых групп креативов в bannerstorage
 * и синхронизирует новые созданные креативы с Директом (добавляет их в Директ)
 */
@Component
@Multilaunch
@Approvers({"mexicano", "hrustyashko", "buhter"})
public class GenerateSmartCreativesOneshot implements SimpleOneshot<InputData, State> {
    private static final Logger logger = LoggerFactory.getLogger(GenerateSmartCreativesOneshot.class);

    // Шаблоны в BannerStorage, для которых можно запустить этот ваншот
    // Здесь просто перечислены все смартовые шаблоны на текущий момент
    // При необходимости список можно расширить
    private static final Set<Integer> SMART_TEMPLATES = Set.of(740, 741, 778, 779, 782, 783, 839, 910, 1051);

    // Макеты в BannerStorage, для которых можно запустить этот ваншот
    // Сейчас здесь перечислены все лейауты смарт-плитки + смарт-ТГО + мозаика
    // При необходимости список можно расширить
    private static final Set<Integer> SMART_LAYOUTS = Set.of(
            44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61);

    // Сколько групп обрабатывать за одну итерацию
    private static final int BATCH_SIZE = 50;

    private final DslContextProvider dslContextProvider;
    private final ClientRepository clientRepository;
    private final AsyncHttpClient asyncHttpClient;
    private final String intapiUrl;
    private final Supplier<String> intapiTvmTicketSupplier;
    private final BannerStorageClientConfiguration bsConfig;

    @Autowired
    public GenerateSmartCreativesOneshot(DslContextProvider dslContextProvider,
                                         ClientRepository clientRepository,
                                         AsyncHttpClient asyncHttpClient,
                                         @Value("${direct_intapi.url}") String intapiUrl,
                                         @Value("${direct_intapi.tvm_app_id}") int intapiTvmAppId,
                                         TvmIntegration tvmIntegration,
                                         @Value("${bannerstorage_api.url}") String bsapiUrl,
                                         @Value("${bannerstorage_api.token}") String bsapiToken,
                                         @Value("${bannerstorage_api.token_file_path}") String bsapiTokenPath) {
        this.dslContextProvider = dslContextProvider;
        this.clientRepository = clientRepository;
        this.asyncHttpClient = asyncHttpClient;
        this.intapiUrl = intapiUrl;
        TvmService dstIntapiService = TvmService.fromIdStrict(intapiTvmAppId);
        this.intapiTvmTicketSupplier = () -> tvmIntegration.getTicket(dstIntapiService);
        this.bsConfig = new BannerStorageClientConfiguration(bsapiUrl, bsapiToken, bsapiTokenPath);
    }

    @Nullable
    @Override
    public State execute(@Nullable InputData inputData, @Nullable State prevState) {
        Preconditions.checkNotNull(inputData);
        if (prevState == null) {
            prevState = new State();
            prevState.setMinCreativeGroupId(0);
        }

        Set<Long> clientIdsBlackList = new HashSet<>(inputData.getClientIdsBlackList());

        List<Integer> groupIds;
        try {
            // Получаем все группы, в которых нет первого макета
            // тут теоретически можно пропустить группы, в которых есть первый макет, но нет остальных
            // но это не критично, а иначе запрос был бы слишком тяжёлым
            groupIds = getGroupsWithoutSpecificLayout(
                    inputData.getTemplateId(), inputData.getLayoutIds().get(0),
                    prevState.getMinCreativeGroupId()
            );
        } catch (RuntimeException e) {
            logger.error("Can't retrieve groups to work with, exit with delay without changes in state", e);
            sleep();
            return prevState;
        }
        if (groupIds.isEmpty()) {
            logger.info("Finished: nothing to do");
            return null;
        }
        //
        for (Integer groupId : groupIds) {
            try {
                // Сначала получим группу
                List<Creative> group = getBannerStorageGroup(groupId);
                Set<Integer> creativeIds = group.stream().map(Creative::getId).collect(toSet());
                Map<Integer, ClientId> creativeClients = getCreativeClients(creativeIds);
                if (creativeClients.isEmpty()) {
                    logger.info("Found BS creative group with no creatives in Direct (it's ok): {}", groupId);
                    continue;
                }
                // Все креативы должны принадлежать одному клиенту
                Preconditions.checkState(creativeClients.values().stream().distinct().count() == 1);

                // Если для этого клиента генерация не требуется -- не будем её делать
                ClientId clientId = creativeClients.values().iterator().next();
                if (clientIdsBlackList.contains(clientId.asLong())) {
                    logger.error("Client {} is in blackList. Skip autogeneration for group: {}", clientId, groupId);
                    continue;
                }

                // Если не все креативы есть в Директе, попытаемся восстановить группу, добавив их в Директ
                // Если не удалось, пропустим эту группу (с выводом в лог)
                int shard = getClientShard(clientId);
                Set<Integer> missingCreatives = Sets.difference(creativeIds, creativeClients.keySet());
                if (!missingCreatives.isEmpty()) {
                    logger.error("Found creatives missing in Direct in group {}: {}",
                            getGroupBrowserUrl(groupId, clientId, shard), toJson(missingCreatives));
                    Set<Integer> existingCreatives = Sets.intersection(creativeIds, creativeClients.keySet());
                    if (existingCreatives.isEmpty()) {
                        logger.error("Can't repair group: {}", groupId);
                        continue;
                    }
                    Integer baseCreativeId = existingCreatives.iterator().next();
                    List<Pair<Integer, Integer>> pairsToAdd = missingCreatives.stream()
                            .map(id -> Pair.of(baseCreativeId, id))
                            .collect(toList());
                    List<SpawnCreativeResult> spawnCreativeResults = addCreativesToDirect(pairsToAdd);
                    List<SpawnCreativeResult> spawnErrors = spawnCreativeResults.stream()
                            .filter(s -> s.getResult() == 0)
                            .collect(toList());
                    if (!spawnErrors.isEmpty()) {
                        logger.error("Can't repair group: {} (errors '{}')", groupId, toJson(spawnErrors));
                        continue;
                    }
                }

                // Добавим новые креативы в BannerStorage
                AddCreativesResult addCreativesResult = addCreativesToBannerStorage(groupId, inputData.getLayoutIds());
                if (addCreativesResult.getError() != null) {
                    logger.error("Failed creative group {}: {}", groupId, addCreativesResult.getError());
                    continue;
                }
                List<AddCreativeResult> withErrors = addCreativesResult.getResults().stream()
                        .filter(res -> res.getError() != null)
                        .collect(toList());
                if (!withErrors.isEmpty()) {
                    logger.warn("Errors adding to BS in group: {} ('{}')", groupId, toJson(withErrors));
                }

                // Успешно созданные креативы добавим в Директ
                List<Pair<Integer, Integer>> creativePairs = addCreativesResult.getResults().stream()
                        .filter(res -> res.getError() == null)
                        .map(res -> Pair.of(res.getBaseCreativeId(), res.getAddedCreativeId()))
                        .collect(toList());
                if (creativePairs.isEmpty()) {
                    logger.warn("No creatives were added, skip group: {}", groupId);
                    continue;
                }
                List<SpawnCreativeResult> spawnCreativeResults = addCreativesToDirect(creativePairs);
                List<SpawnCreativeResult> spawnErrors = spawnCreativeResults.stream()
                        .filter(s -> s.getResult() == 0)
                        .collect(toList());
                if (!spawnErrors.isEmpty()) {
                    logger.error("Errors adding to Direct in group: {} ('{}')", groupId, toJson(spawnErrors));
                }
                List<Integer> addedCreativeIds = spawnCreativeResults.stream()
                        .filter(s -> s.getResult() != 0)
                        .map(SpawnCreativeResult::getChild)
                        .collect(toList());
                if (!addedCreativeIds.isEmpty()) {
                    logger.info("Added creatives to direct: {}", toJson(addedCreativeIds));
                }

                // Запишем в лог ссылку, по которой можно быстро её открыть под супером или супер-ридером
                logger.info("Processed group: {}", getGroupBrowserUrl(groupId, clientId, shard));
                //
            } catch (RuntimeException e) {
                // Продолжаем выполнение, но пометим проблемную группу в логе,
                // чтобы можно было потом легко грепнуть группы с ошибками в случае чего
                logger.error("Failed creative group: " + groupId, e);
            }
        }
        logger.info("Processed {} creative groups", groupIds.size());
        return new State()
                .withMinCreativeGroupId(1 + groupIds.get(groupIds.size() - 1));
    }

    private Map<Integer, ClientId> getCreativeClients(Collection<Integer> creativeIds) {
        return dslContextProvider.ppcdict()
                .select(SHARD_CREATIVE_ID.CREATIVE_ID,
                        SHARD_CREATIVE_ID.CLIENT_ID)
                .from(SHARD_CREATIVE_ID)
                .where(SHARD_CREATIVE_ID.CREATIVE_ID.in(creativeIds))
                .fetchMap(record -> record.get(SHARD_CREATIVE_ID.CREATIVE_ID).intValue(),
                        record -> ClientId.fromLong(record.get(SHARD_CREATIVE_ID.CLIENT_ID)));
    }

    private int getClientShard(ClientId clientId) {
        return dslContextProvider.ppcdict()
                .select(SHARD_CLIENT_ID.SHARD)
                .from(SHARD_CLIENT_ID)
                .where(SHARD_CLIENT_ID.CLIENT_ID.eq(clientId.asLong()))
                .fetchOne(SHARD_CLIENT_ID.SHARD)
                .intValue();
    }

    private String getGroupBrowserUrl(int groupId, ClientId clientId, int shard) {
        String login = dslContextProvider.ppc(shard)
                .select(USERS.LOGIN)
                .from(USERS)
                .where(USERS.CLIENT_ID.eq(clientId.asLong()))
                .and(USERS.REP_TYPE.eq(UsersRepType.chief))
                .fetchOne(USERS.LOGIN);
        String baseUrl;
        switch (Environment.getCached()) {
            case PRODUCTION:
                baseUrl = "https://direct.yandex.ru";
                break;
            case TESTING:
                baseUrl = "https://test-direct.yandex.ru";
                break;
            case DEVTEST:
            case DEVELOPMENT:
                baseUrl = "https://8080.beta1.direct.yandex.ru";
                break;
            default:
                baseUrl = "https://8080.beta1.direct.yandex.ru";
                break;
        }
        return baseUrl + String.format("/smart-designer/%d/edit/?client_login=%s", groupId, login);
    }

    private List<Creative> getBannerStorageGroup(int groupId) {
        RequestBuilder requestBuilder = new RequestBuilder("GET");
        requestBuilder.setUrl(bsConfig.getUrl() +
                String.format("/creativeGroups/%d?include=creatives.layoutCode", groupId)
        )
                .setCharset(StandardCharsets.UTF_8)
                .setHeader(ACCEPT, "application/json")
                .setHeader(AUTHORIZATION, "OAuth: " + bsConfig.getToken());
        //
        Response response = executeRequest(requestBuilder);
        //
        String responseBody = response.getResponseBody();
        int statusCode = response.getStatusCode();
        logger.info("Got response: status {} body '{}'", statusCode, responseBody);
        if (statusCode != 200) {
            throw new RuntimeException("Got status code " + statusCode);
        }
        CreativeGroupDTO result = fromJson(responseBody, CreativeGroupDTO.class);
        return result.getCreatives();
    }

    private List<Integer> getGroupsWithoutSpecificLayout(int templateId, int layoutId, int minGroupId) {
        RequestBuilder requestBuilder = new RequestBuilder("GET");
        requestBuilder.setUrl(bsConfig.getUrl() +
                String.format(
                        "/creativeGroups/withoutLayout/%d?templateId=%d&minGroupId=%d&limit=%d",
                        layoutId, templateId, minGroupId, BATCH_SIZE
                ))
                .setCharset(StandardCharsets.UTF_8)
                .setHeader(ACCEPT, "application/json")
                .setHeader(AUTHORIZATION, "OAuth: " + bsConfig.getToken());
        //
        Response response = executeRequest(requestBuilder);
        //
        String responseBody = response.getResponseBody();
        int statusCode = response.getStatusCode();
        logger.info("Got response: status {} body '{}'", statusCode, responseBody);
        if (statusCode != 200) {
            throw new RuntimeException("Got status code " + statusCode);
        }
        GroupIdsDTO result = fromJson(responseBody, GroupIdsDTO.class);
        return result.getItems().stream()
                .map(GroupIdDTO::getId)
                .collect(toList());
    }

    private AddCreativesResult addCreativesToBannerStorage(int groupId, List<Integer> layoutIds) {
        RequestBuilder requestBuilder = new RequestBuilder("POST");
        requestBuilder.setUrl(bsConfig.getUrl() +
                String.format(
                        "/creativeGroups/%d/addCreatives?layoutIds=%s",
                        groupId, layoutIds.stream().map(Object::toString).collect(joining(","))
                ))
                .setCharset(StandardCharsets.UTF_8)
                .setHeader(ACCEPT, "application/json")
                .setHeader(AUTHORIZATION, "OAuth: " + bsConfig.getToken());
        //
        Response response = executeRequest(requestBuilder);
        //
        String responseBody = response.getResponseBody();
        int statusCode = response.getStatusCode();
        logger.info("Got response: status {} body '{}'", statusCode, responseBody);
        if (statusCode != 201) {
            throw new RuntimeException("Got status code " + statusCode);
        }
        return fromJson(responseBody, AddCreativesResult.class);
    }

    private List<SpawnCreativeResult> addCreativesToDirect(List<Pair<Integer, Integer>> creativePairs) {
        RequestBuilder requestBuilder = new RequestBuilder("POST");
        String requestBody = String.format(
                "{\"method\":\"spawn_creatives\",\"params\":{\"creatives_pairs\":\"%s\"}}",
                creativePairs.stream()
                        .map(pair -> String.format("%d:%d", pair.getLeft(), pair.getRight()))
                        .collect(joining(";"))
        );
        requestBuilder.setUrl(intapiUrl + "/jsonrpc/BsFront")
                .setCharset(StandardCharsets.UTF_8)
                .setBody(requestBody)
                .setHeader(CONTENT_TYPE, "application/json")
                .setHeader(ACCEPT, "application/json")
                .setHeader(SERVICE_TICKET_HEADER, intapiTvmTicketSupplier.get());
        //
        Response response = executeRequest(requestBuilder);
        //
        String responseBody = response.getResponseBody();
        int statusCode = response.getStatusCode();
        logger.info("Got response: status {} body '{}'", statusCode, responseBody);
        if (statusCode != 200) {
            throw new RuntimeException("Got status code " + statusCode);
        }
        SpawnCreativesResult result = fromJson(responseBody, SpawnCreativesResult.class);
        return result.getResults();
    }

    private Response executeRequest(RequestBuilder requestBuilder) {
        Request request = requestBuilder
                .setRequestTimeout(60_000)
                .setReadTimeout(60_000)
                .build();
        if (request.getStringData() != null) {
            logger.info("Sending request {}, body '{}'", request.toString(), request.getStringData());
        } else {
            logger.info("Sending request {}", request.toString());
        }
        ListenableFuture<Response> future = asyncHttpClient.executeRequest(request);
        try {
            return future.get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (ExecutionException e) {
            throw new RuntimeException("Can't get the result", e);
        }
    }

    private void sleep() {
        try {
            Thread.sleep(60_000);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    @Override
    public ValidationResult<InputData, Defect> validate(InputData inputData) {
        ItemValidationBuilder<InputData, Defect> vb = ItemValidationBuilder.of(inputData);
        vb.item(inputData.getTemplateId(), "templateId")
                .check(fromPredicate(in(SMART_TEMPLATES), CommonDefects.invalidValue()));
        vb.list(inputData.getLayoutIds(), "layoutIds")
                .check(fromPredicateOfNullable(notNull(), CommonDefects.notNull()))
                .check(fromPredicate(notEmptyCollection(), CollectionDefects.notEmptyCollection()))
                .checkEach(fromPredicate(in(SMART_LAYOUTS), CommonDefects.invalidValue()));
        vb.list(inputData.getClientIdsBlackList(), "clientIdsAllowList")
                .check(fromPredicateOfNullable(notNull(), CommonDefects.notNull()));
        return vb.getResult();
    }
}
